/*
 * MX28 GPIO access
 *
 * Iwo Mergler
 */
#ifndef _MX28_GPIO_H
#define _MX28_GPIO_H

/* The regspinctrl.h file is silly, it contains more lines of code than
 * there are total bits within the registers it describes. Shrinking
 * 1000 lines of code now:
 */

#include <asm/io.h>

/* PIN code */
#define PIN(bank,bit)     ((bank)*32+(bit))

#define PINCTRL_BASE            (0x80018000)

/* Register block (reg, set, clr, toggle) modifiers */
#define R   +(0x0)
#define SET +(0x4)
#define CLR +(0x8)
#define TOG +(0xc)

/* PINCTRL Register block addresses */
#define PINCTRL_CTRL            (PINCTRL_BASE)
#define PINCTRL_MUXSEL(n)       (PINCTRL_BASE + 0x0100 + (n)*0x10) /* n max 13 */
#define PINCTRL_DRIVE(n)        (PINCTRL_BASE + 0x0300 + (n)*0x10) /* n max 19 */
#define PINCTRL_PULL(n)         (PINCTRL_BASE + 0x0600 + (n)*0x10) /* n max 4 + 2 keeper */
#define PINCTRL_DOUT(n)         (PINCTRL_BASE + 0x0700 + (n)*0x10) /* n max 4 */
#define PINCTRL_DIN(n)          (PINCTRL_BASE + 0x0900 + (n)*0x10) /* n max 4 */
#define PINCTRL_DOE(n)          (PINCTRL_BASE + 0x0B00 + (n)*0x10) /* n max 4 */
#define PINCTRL_PIN2IRQ(n)      (PINCTRL_BASE + 0x1000 + (n)*0x10) /* n max 4 */
#define PINCTRL_IRQEN(n)        (PINCTRL_BASE + 0x1100 + (n)*0x10) /* n max 4 */
#define PINCTRL_IRQLEVEL(n)     (PINCTRL_BASE + 0x1200 + (n)*0x10) /* n max 4 */
#define PINCTRL_IRQPOL(n)       (PINCTRL_BASE + 0x1300 + (n)*0x10) /* n max 4 */
#define PINCTRL_IRQSTAT(n)      (PINCTRL_BASE + 0x1400 + (n)*0x10) /* n max 4 */
#define PINCTRL_EMI_ODTCTRL     (PINCTRL_BASE + 0x1A40)
#define PINCTRL_EMI_DS_CTRL     (PINCTRL_BASE + 0x1B80)

enum muxmode { MUX0=0, MUX1, MUX2, IN, OUT };

/* Set consecutive bits to specified value (right justified) in reg */
static inline void regset(const unsigned a, const unsigned v, const unsigned shift, const unsigned bits)
{
	unsigned mask = ((1 << bits)-1);
	writel((v & mask) << shift, a SET);
	writel((~v & mask) << shift, a CLR);
}

static inline unsigned regget(const unsigned a, const unsigned shift, const unsigned bits)
{
	unsigned mask = ((1 << bits)-1);
	return (readl(a R) >> shift) & mask;
}

static inline void gpio_mux(const unsigned pin, const enum muxmode mux)
{
	writel(0x40000000, PINCTRL_CTRL CLR); /* Make sure the clock is on */
	if (mux < IN) {
		/* Other function - mux registers have 2 bits per pin */
		regset(PINCTRL_MUXSEL(pin/16), mux, (pin%16)*2, 2);
	} else {
		/* GPIO mode. */
		regset(PINCTRL_MUXSEL(pin/16), 0x3, (pin%16)*2, 2);
		if (mux==OUT)
			regset(PINCTRL_DOE(pin/32), 1, pin%32, 1);
		else
			regset(PINCTRL_DOE(pin/32), 0, pin%32, 1);
	}
}

/* Not all pins support this. volt 0=1.8V, 1=3.3V, amp 0=4mA, 1=8mA, 2=12mA. */
static inline void gpio_drive(const unsigned pin, const unsigned volt, const unsigned amp)
{
	unsigned drv = ((volt << 2) | amp) & 0x7;
	writel(0x40000000, PINCTRL_CTRL CLR); /* Make sure the clock is on */
	regset(PINCTRL_DRIVE(pin/8), drv, (pin%8) * 4, 4);
}

/* Not all pins support this. 0=off, 1=on */
static inline void gpio_pullup(const unsigned pin, const unsigned yes)
{
	regset(PINCTRL_PULL(pin/32), !!yes, pin%32, 1);
}

static inline void gpio_set(const unsigned pin, const unsigned v)
{
	regset(PINCTRL_DOUT(pin/32), !!v, pin%32, 1);
}

static inline void gpio_toggle(const unsigned pin)
{
	writel(1<<(pin%32), PINCTRL_DOUT(pin/32) TOG);
}

static inline unsigned gpio_read(const unsigned pin)
{
	return readl(PINCTRL_DIN(pin/32)) & (1<<(pin%32));
}

static inline void gpio_init(void)
{
	writel(0x40000000, PINCTRL_CTRL CLR); /* Make sure the clock is on */
}

/* Full pin state manipulation */

struct pininfo {
	unsigned muxmode : 4;  /* 0, 1, 2, in, out */
	unsigned volt : 4;     /* 0, 1 */
	unsigned amp  : 4;     /* 0, 1, 2 */
	unsigned pullup : 4;   /* 0, 1 */
	unsigned dout  : 4;    /* 0, 1 (value driven by GPIO) */
	unsigned din : 4;      /* 0, 1 */
};

/* Useful macros */
#define MM(n) .muxmode = n
#define M_IN .muxmode = IN
#define M_OUT .muxmode = OUT
#define V18 .volt = 0
#define V33 .volt = 1
#define A04 .amp = 0
#define A08 .amp = 1
#define A12 .amp = 2
#define PU .pullup = 1
#define DO(n) .dout = n

struct gpio_preset {
	unsigned pin;
	struct pininfo v;
};

#define NOPIN ((unsigned)(-1))
#define pin(b,i) .pin = PIN(b,i)

static inline void get_pininfo(const unsigned pin, struct pininfo *p)
{
	unsigned muxsel = regget(PINCTRL_MUXSEL(pin/16), (pin%16)*2, 2);
	unsigned doe = regget(PINCTRL_DOE(pin/32), pin%32, 1);
	unsigned drv = regget(PINCTRL_DRIVE(pin/8), (pin%8)*4, 4);
	unsigned pull = regget(PINCTRL_PULL(pin/32), pin%32, 1);
	unsigned dout = regget(PINCTRL_DOUT(pin/32), pin%32, 1);
	unsigned din = regget(PINCTRL_DIN(pin/32), pin%32, 1);

	if (muxsel == 0x3) {
		if (doe)
			p->muxmode = OUT;
		else
			p->muxmode = IN;
	} else {
		p->muxmode = muxsel;
	}

	p->volt = drv >> 2;
	p->amp = drv & 0x3;
	p->pullup = pull;
	p->dout = dout;
	p->din = din;
}

static inline void set_pininfo(const unsigned pin, const struct pininfo *p)
{
	gpio_set(pin, p->dout);
	gpio_mux(pin, IN);
	gpio_drive(pin, p->volt, p->amp);
	gpio_pullup(pin, p->pullup);
	gpio_mux(pin, p->muxmode);
}

#endif /* _MX28_GPIO_H */
