/*
 * GPIO command line support for MX28 series processors.
 *
 * Iwo Mergler <Iwo.Mergler@netcommwireless.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <command.h>
#include <asm/byteorder.h>

#include <asm/arch/imx-regs.h>
#include <exports.h>

#include "gpio.h"


/* Set gpio muxmode. Args: bank bit mux0|mux1|mux2|in|out */
static int fun_gpiox(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	enum muxmode mux;
	unsigned pin;

	if (argc != 4) return -1;

	if      (strcmp("mux0",argv[3])==0) mux=MUX0;
	else if (strcmp("mux1",argv[3])==0) mux=MUX1;
	else if (strcmp("mux2",argv[3])==0) mux=MUX2;
	else if (strcmp("in",argv[3])==0) mux=IN;
	else if (strcmp("out",argv[3])==0) mux=OUT;
	else {
		printf("Bad muxmode: %s\r\n", argv[3]);
		return -1;
	}

	pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));

	gpio_mux(pin, mux);

	return 0;
}
U_BOOT_CMD(gpiox, CONFIG_SYS_MAXARGS, 1, fun_gpiox,
	"GPIO multiplexer control",
	"BANK BIT mux0|mux1|mux2|in|out\n"
);

/* Set gpio drive mode. Args: bank pin volt amp */
static int fun_gpiod(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	unsigned pin;
	if (argc != 5) return -1;

	pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));

	gpio_drive(pin, simple_strtol(argv[3], NULL, 0), simple_strtol(argv[4], NULL, 0));
	return 0;
}
U_BOOT_CMD(gpiod, CONFIG_SYS_MAXARGS, 1, fun_gpiod,
	"GPIO PAD drive control",
	"BANK BIT VOLT AMP\n"
	"	VOLT: 0=1.8V, 1=3.3V\n"
	"	AMP: 0=4mA, 1=8mA, 2=12mA\n"
);


/* Set pull-up. Args: bank pin pu */
static int fun_gpiop(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	unsigned pin;
	if (argc != 4) return -1;

	pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));

	gpio_pullup(pin, simple_strtol(argv[3], NULL, 0));
	return 0;
}
U_BOOT_CMD(gpiop, CONFIG_SYS_MAXARGS, 1, fun_gpiop,
	"GPIO pull-up control",
	"BANK BIT PU\n"
	"	PU: 0=off, 1=on\n"
);


/* GPIO read */
static int fun_gpior(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	static int pin = -1;
	char* bank_pin_param;
	char* p;
	
	int bank;
	int bit;
	
	int pin_valid;

	/* use the previous pin if repeated */
	if(flag & CMD_FLAG_REPEAT) {
		
		/* set valid flag */
		pin_valid=1;
	}
	/* gpior bank pin - backward compatibility */
	else if(argc==3) {
		pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));
		
		/* set valid flag */
		pin_valid=1;
	}
	/* gpior bank:bit */
	else if(argc==2) {
		bank_pin_param=strdup(argv[1]);
		
		/* get bank */
		if( (p = strtok(bank_pin_param, ":"))!=NULL ) {
			bank=simple_strtoul(p,NULL,0);
			
			/* get bit */
			if( (p = strtok(NULL, ":"))!=NULL ) {
				bit=simple_strtoul(p,NULL,0);
				
				pin = PIN(bank, bit);
				
				/* set valid flag */
				pin_valid=1;
			}
		}
		
		free(bank_pin_param);
	}
	else {
		pin_valid=0;
	}
			
	/* bypass if not valid */
	if (!pin_valid) return -1;

	if (pin >= 0) {
		//gpio_mux(pin, IN);
		if (gpio_read(pin)) {
			printf("1\r\n");
			return 0;
		} else {
			printf("0\r\n");
			return 1;
		}
	} else {
		printf("read bank/pin first before repeat\r\n");
	}
	return 0;
}
U_BOOT_CMD(gpior, CONFIG_SYS_MAXARGS, 1, fun_gpior,
	"GPIO read",
	"BANK BIT [repeatable]\n"
	"gpior BANK:BIT [repeatable]"
);


/* GPIO write */
static int fun_gpiow(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	unsigned pin;
	if (argc != 4) return -1;

	pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));

	gpio_mux(pin, OUT);
	gpio_set(pin, simple_strtol(argv[3], NULL, 0));
	return 0;
}
U_BOOT_CMD(gpiow, CONFIG_SYS_MAXARGS, 1, fun_gpiow,
	"GPIO write",
	"BANK BIT VALUE\n"
);


static int fun_gpiot(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	static int pin = -1;

	if (argc>1 && !(flag & CMD_FLAG_REPEAT)) {
		pin = PIN(simple_strtoul(argv[1],NULL,0), simple_strtoul(argv[2],NULL,0));
	}

	if (argc != 0 && argc != 3) return -1;

	if (pin > 0) {
		gpio_toggle(pin);
	} else {
		printf("read bank/pin first before repeat\r\n");
	}
	return 0;
}
U_BOOT_CMD(gpiot, CONFIG_SYS_MAXARGS, 1, fun_gpiot,
	"GPIO toggle",
	"BANK BIT, can be repeated\n"
);

static void binprint(uint32_t v)
{
	unsigned i;
	for (i=0; i<32; i++) {
		if (v & 0x80000000)
			printf("1");
		else
			printf("0");
		if (i % 4 == 3) printf(" ");
		v <<= 1;
	}
}

static void regbank(unsigned base, unsigned regs)
{
	while (regs--) {
		unsigned v = readl(base);
		printf("[%08x] %08x ", base, v);
		binprint(v);
		printf("\n");
		base += 0x10;
	}
}

static int fun_gpiodump(cmd_tbl_t *cmdtp, int flag, int argc, char *const argv[])
{
	struct pininfo p;
	int bank, bit;
	int col;
	char mux;

	if (0) {
		regbank(PINCTRL_CTRL, 1);
		regbank(PINCTRL_MUXSEL(0), 14);
		regbank(PINCTRL_DRIVE(0), 20);
		regbank(PINCTRL_PULL(0), 7);
		regbank(PINCTRL_DOUT(0), 5);
		regbank(PINCTRL_DIN(0), 5);
		regbank(PINCTRL_DOE(0), 5);
	}

	printf("PINCTRL_CTRL=%08x\n", readl(PINCTRL_CTRL));

	printf("\nPIM(B,bi)=MVAPOI - Mux(0,1,2,i,o), Volt, Amp, Pull-up, dOut, dIn\n\n");

	for (bank=0; bank<5; bank++) {
		col=0;
		for (bit=31; bit>=0; bit--) {
			get_pininfo(PIN(bank, bit), &p);
			switch (p.muxmode) {
				case IN:  mux='i'; break;
				case OUT: mux='o'; break;
				default: mux='0'+p.muxmode; break;
			}
			printf("PIN(%d,%2d)=%c%d%d%c%d%d ",
				bank, bit, mux, p.volt, p.amp, p.pullup?'P':'.', p.dout, p.din);

			col++;
			if (col==4) {
				col = 0;
				printf("\n");
			}
		}
		printf("\n");
	}

	return 0;
}
U_BOOT_CMD(gpiodump, CONFIG_SYS_MAXARGS, 1, fun_gpiodump,
	"Dump GPIO registers",
	""
);
