/*
Microscopic printf implementation.

Iwo Mergler <Iwo.Mergler@netcommwireless.com>
*/
#include <common.h>
#include <asm/gpio.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/iomux-mx28.h>
#include <asm/arch/clock.h>
#include <asm/arch/sys_proto.h>
#include <stdarg.h>

DECLARE_GLOBAL_DATA_PTR;

static unsigned base = DUART_BASE;

static void debugchar(char ch)
{
	send_app_uart((void*)base, ch);
}

static void print(const char *s)
{
	while (*s) {
		debugchar((uint8_t)*s);
		s++;
	}
}

static inline char hexchar(unsigned v)
{
	return ((v>9)?'a'+v-10:'0'+v);
}

static void phex8(uint8_t x)
{
	debugchar(hexchar(x >> 4));
	debugchar(hexchar(x & 0xf));
}

static void phex16(uint16_t x)
{
	phex8(x >> 8);
	phex8(x & 0xff);
}

static void phex32(uint32_t x)
{
	phex16(x >> 16);
	phex16(x & 0xffff);
}

static void printint(int i)
{
	unsigned int q;
	int d=10000;
	if (i<0) { debugchar('-'); i=-i; }
	/* skip leading zeros */
	q=i/d;
	while ((q==0) && (d>=10)) {
		d=d/10;
		q=i/d;
	};
	while (d>0) {
		debugchar('0'+(char)q);
		i -= q*d;
		d=d/10;
		q=i/d;
	}
}

void app_uartbase(unsigned base_addr)
{
	base = base_addr;
}

/* Formatted hex strings must be %2x, %4x or %8x. Default is %4x */
/* float/double are printed with specified number of fractional digits, default 2 */
void app_printf(const char *fmt, ...)
{
	uint8_t chars=0;
	char c;
	va_list ap;
	va_start(ap, fmt);

	c = *fmt;
	while (c) {
		if (c == '%') {
			chars = 0;
			while(1) {
				fmt++; c = *fmt;
				switch (c) {
					case 'd' :
					case 'u' :
						printint(va_arg(ap, int));
						break;
					case 'x' :
						switch (chars) {
							case 2:
								phex8((uint8_t)va_arg(ap, int));
								break;
							case 8:
								phex32(va_arg(ap, uint32_t));
								break;
							default:
								phex16((uint16_t)(va_arg(ap, int)));
								break;
						}
						break;
					case 'p' :
						phex32(va_arg(ap, uint32_t));
						break;
					case 's' :
						print(va_arg(ap, const char *));
						break;
					case 'c' :
						debugchar((uint8_t)va_arg(ap, int));
						break;
					case '.' :
						chars = 0;
						continue;
						break;
					default :
						if (c>='0' || c<='9') {
							chars = chars*10+(c-'0');
							continue;
						} else {
							print("??");
						}
						break;
				}
				goto get_out;
			};
get_out:;
		} else {
			debugchar(c);
		}
		fmt++; c = *fmt;
	}
	va_end(ap);
}
