/* Microscopic display driver library. Currently supports
 * the SH1106 OLED display controller in 4-wire SPI mode,
 * as integrated on the SAS1-09046 (128x64) OLED display panel.
 *
 * Level0: SPI master (write only) bitbashing implementation
 * Level1: SH1106 display controller support
 * Level2: SAS1-09046 panel support
 * Level3: Frame buffer
 * Level4: Graphics primitives
 * Level5: Silly toys
 *
 * Iwo.Mergler@netcommwireless.com
 */
#ifndef _DISPLAY_H
#define _DISPLAY_H


/*
 * Level0, bitbashing SPI master, mode 0, write only
 */

/* Display pin assignment, NTC-20 specific. Requires
 * pins to already be configured as GPIO outputs
 */
#define SCL_DSP PIN(2,10)
#define SDI_DSP PIN(2,8)
#define CSN_DSP PIN(2,3)
#define D_C_DSP PIN(2,20)
#define RST_DSP PIN(0,17)

/* This is a function of display internal wiring. Current
 * display panel has two pixel columns wrapped */
#define DSP_HWRAP 2


/*
 * Level1, SH1106 display controller support
 */

/* Send command byte to display */
void dsp_cmd(uint8_t v);

/* Write a byte to display, advance column and page pointers.
 * Start with *ctx==0, after that it is used to maintain context.
 * The sequence is not reentrant. */
void dsp_advance(unsigned *ctx, uint8_t v);

/* Whole framebuffer data write, automatically advances
 * page pointer when needed. Starts at (0,0). May be short. */
void dsp_dta(int n, uint8_t *data);

/* Whole framebuffer data write, automatically advances
 * page pointer when needed. */
void dsp_repeat(int n, uint8_t v);

/* Display is organised into pages (horizontal 8-pixel stripes).
 * A column is one pixel wide. Display framebuffer is accessed
 * by writing a page, column address auto-increments. A data
 * byte represents a 8-pixel column, LSB on top. */

#define DSP_LCOLA(a)   (0x00 | ((a) & 0xF))         /* Set low column address */
#define DSP_HCOLA(a)   (0x10 | (((a) >> 4) & 0xF))  /* Set high column address */
#define DSP_CVOLT(v)   (0x30 | ((v) & 0x3))         /* Set charge pump voltage */
	#define CV64 0
	#define CV74 1
	#define CV80 2
	#define CVON 2
	#define CV90 3
#define DSP_LINEA(a)   (0x40 | ((a) & 0x3F))        /* Set display start line (pixel), for smooth scroll */ 
#define DSP_CTRST0     (0x81)                       /* Set contrast. Must be followed by CTRST1 */
#define DSP_CTRST1(v)  (v)                          /* Contrast value, 0-255 */
#define DSP_SEGMAP(b)  (0xA0 | ((b) & 0x01))        /* Set column map to 0-normal, 1-reverse. Display specific */
#define DSP_SWITCH(b)  (0xA4 | ((b) & 0x01))        /* Display driver off/on. Frame buffer is preserved */
#define DSP_REVERSE(b) (0xA6 | ((b) & 0x01))        /* Invert display (negative) */
#define DSP_MUXR0      (0xA8)                       /* Set Multiplex ratio. Followed by MUXR1 */
#define DSP_MUXR1(v)   ((v) & 0x3F)                 /* Multiplexer ratio value, 0-3f */
#define DSP_DCDC0      (0xAD)                       /* Enable/disable DC/DC. Panel must be off */
#define DSP_DCDC1(b)   (0x8A | ((b) & 0x01))        /* Enable/disable DC/DC */
#define DSP_LPOW(b)    (0xAE | ((b) & 0x01))        /* Low power mode (display off, framebuffer on) */
#define DSP_PAGEA(a)   (0xB0 | ((a) & 0x07))        /* Set page address for framebuffer access */
#define DSP_COMSCAN(b) (0xC0 | (((b) & 0x01) << 3)) /* Set common scan dir, panel specific */
#define DSP_DOFF0      (0xD3)                       /* Set display offset. Panel dependent. */
#define DSP_DOFF1(v)   ((v) & 0x3F)                 /* display offset to rotate COM lines */
#define DSP_FREQ0      (0xD5)                       /* Set clock divider - affects frame scan rate */
#define DSP_FREQ1(v)   (v)                          /* DIV(n) | FRQ(n) */
	#define DIV(n) (((n)-1) & 0xF)              /* Divider value, 1-16 */
	#define FRQ(n) (((n) & 0x0F) << 4)          /* oscillator adjust, -25%, -20%, ..., +45%, +50% */
	#define FRQD   FRQ(5)                       /* Default oscillator adjust, 0% */
#define DSP_DCPC0      (0xD9)                       /* Set discharge / precharge period */
#define DSP_DCPC1(v)   (v)                          /* PRECH(n) | DISCH(n) */
	#define PRECH(n) ((n) & 0x0F)               /* Precharge timing, 1-15 clocks */
	#define PRECHD   (2)                        /* Default precharge, 2 clocks */
	#define DISCH(n) (((n) & 0x0F) << 4)        /* Discharge timing, 1-15 clocks */
	#define DISCHD   (2 << 4)                   /* Default discharge, 2 clocks */
#define DSP_CPADS0     (0xDA)                       /* Common pads config, panel specific. */
#define DSP_CPADS1(b)  (0x02 | (((b) & 0x01) << 4)) /* 0-sequential, 1-alternate (default) */
#define DSP_VCOM0      (0xDB)                       /* Set VCOM deselect voltage level */
#define DSP_VCOM1(v)   (v)                          /* 0-1F = 0.430, 20-3F = 0.770 (default), 40-ff = 1 */
#define DSP_RMWS       (0xE0)                       /* Start read-modify-write command */
#define DSP_RMWE       (0xEE)                       /* End read-modify-write command */
#define DSP_NOP        (0xE3)                       /* No operation command */


/*
 * Level2: SAS1-09046 panel support
 */

/* Reset and configure display to be enabled, erased and ready to use. Must be called. */
int dsp_init(void);


/*
 * Level3: Framebuffer
 *
 * This framebuffer is organised as described above (8-pixel strips/pages, 1-pixel columns,
 * 1 byte is a 1x8 pixel column, LSB on top.
 *
 * A whole frame buffer is 1K in size. If this is a problem, a single-page buffer
 * is also possible, but requires the drawing to happen multiple times. We don't support
 * that atm.
 */
#define DSP_RESX (128)
#define DSP_RESY (64)
#define DSP_PAGES (DSP_RESY/8)

#define DSP_COLUMN(x,y) (x)
#define DSP_PAGE(x,y)   ((y) / 8)
#define DSP_BYTE(x,y)   (DSP_PAGE((x),(y)) * DSP_RESX + DSP_COLUMN((x),(y)))
#define DSP_BIT(x,y)    ((y) % 8)

extern uint8_t dsp_FB[DSP_RESX * DSP_PAGES];

/* Push whole frame buffer out to display */
void dsp_flush(void);


/*
 * Level4: Graphics primitives, operate on frame buffer.
 */

#define DSP_SETPIXEL(x,y) dsp_FB[DSP_BYTE((x),(y))] |= (1 << DSP_BIT((x),(y)))
#define DSP_CLRPIXEL(x,y) dsp_FB[DSP_BYTE((x),(y))] &= ~(1 << DSP_BIT((x),(y)))

#define DSP_PIXEL(x,y,c) do{ if (c) DSP_SETPIXEL((x),(y)); else DSP_CLRPIXEL((x),(y)); }while(0)

#define DSP_RPIXEL(x,y) (!!(dsp_FB[DSP_BYTE((x),(y))] & (1 << DSP_BIT((x),(y)))))

#define DSP_FILL(c) memset(dsp_FB, (c) ? 0xFF : 0x00, sizeof(dsp_FB))

void dsp_line(int x1, int y1, int x2, int y2, int c);

void dsp_vline(int x, int y1, int y2, int c);

void dsp_hline(int x1, int x2, int y, int c);

void dsp_square(int x1, int y1, int x2, int y2, int c);

/*
 * Level5: Silly toys
 */

/* Fills framebuffer with random pixels. Alternate
 * with flush calls to create snowstorm */
void dsp_snow(void);

/* Wrapped Conway's Game of Life. Single step. */
void dsp_life(void);

/* Pong, single step.
 * controllers: 2 AI, -1 up, 0 stay, 1 down
 * Returns -1 after miss, resets to center. */
int dsp_pong(int left, int right);

#endif /* _DISPLAY_H */
