#include <common.h>
#include <config.h>


#include <asm/io.h>
#include <asm/arch/iomux-mx28.h>

#include "gpio.h"
#include "display.h"

/* Bitbashing SPI, Mode 0. Display samples on rising clock edge.
 * Datasheet says 200ns setup / hold time, 500ns cycle. We're at
 * 1us / cycle, full out. */
static void tx_byte(uint8_t v)
{
	int i;
	for (i = 0; i < 8; i++) {
		gpio_set(SDI_DSP, v & 0x80);
		gpio_set(SCL_DSP, 0);
		/* Symmetry delay */
		gpio_set(SDI_DSP, v & 0x80);
		v <<= 1;
		gpio_set(SCL_DSP, 1);
	}
}

void dsp_cmd(uint8_t v)
{
	gpio_set(D_C_DSP, 0); /* Command mode */
	gpio_set(CSN_DSP, 0); /* Chip select */
	tx_byte(v);
	gpio_set(CSN_DSP, 1); /* Chip select off */
}

/* Write a byte to framebuffer, advance column and page pointers.
 * Start with *ctx==0, after it is used to maintain context. Only
 * one sequence is allowed concurrently. */
void dsp_advance(unsigned *ctx, uint8_t v)
{
	unsigned c,p;
	if (!ctx) {
		c = 0;
		p = 0;
	} else {
		c = *ctx & 0xFFFF;
		p = *ctx >> 16;
	}

	if (c == 0) {
		gpio_set(CSN_DSP, 1); /* Chip select off */
		/* Set pointers, column address is auto incrementing,
		 * page address isn't. */
		dsp_cmd(DSP_PAGEA(p));
		dsp_cmd(DSP_LCOLA(DSP_HWRAP));
		dsp_cmd(DSP_HCOLA(DSP_HWRAP));
		mdelay(1);
		gpio_set(D_C_DSP, 1); /* Data mode */
		gpio_set(CSN_DSP, 0); /* Chip select on */
	}

	tx_byte(v);

	c++;
	if (c >= DSP_RESX) {
		c = 0;
		p++;
	}

	*ctx = (p << 16) | c;
}

/* Whole framebuffer data write, automatically advances
 * page pointer when needed */
void dsp_dta(int n, uint8_t *data)
{
	int i;
	unsigned ctx = 0;

	for (i = 0; i < n; i++) {
		dsp_advance(&ctx, data[i]);
	}

}

/* Whole framebuffer data write, automatically advances
 * page pointer when needed. */
void dsp_repeat(int n, uint8_t v)
{
	int i;
	unsigned ctx = 0;
	for (i = 0; i < n; i++) {
		dsp_advance(&ctx, v);
	}
}

int dsp_init(void)
{
	/* Preset GPIO values */
	gpio_set(RST_DSP, 0); /* Assert reset */
	gpio_set(SCL_DSP, 0);
	gpio_set(SDI_DSP, 0);
	gpio_set(CSN_DSP, 1);
	gpio_set(D_C_DSP, 0);
	udelay(10);
	gpio_set(RST_DSP, 1); /* Release reset */
	udelay(3);

	/* Startup sequence. Commented out items are those where the defaults
	 * are already set correctly. */
	{
		const uint8_t V[] = {

		DSP_LPOW(0), /* display off */

		DSP_LCOLA(0),
		DSP_HCOLA(0), /* Column address */

		DSP_LINEA(0), /* view to line 0 */

		DSP_PAGEA(0), /* Page address */

		DSP_CTRST0,
		DSP_CTRST1(127), /* Set contrast */

//		DSP_SEGMAP(1), /* Segment map to reverse (left right swap) */

		DSP_REVERSE(0), /* Normal (positive) display */

//		DSP_MUXR0,
//		DSP_MUXR1(0x3F), /* Multiplex ratio, 1/64 */

		DSP_DCDC0,
		DSP_DCDC1(1), /* Charge pump on */

		DSP_CVOLT(CV80), /* 8.0 V */

//		DSP_COMSCAN(1), /* com scan direction (top bottom swap) */

		DSP_DOFF0,
		DSP_DOFF1(0), /* com pad rotation (vertical display offset) */

//		DSP_FREQ0,
//		DSP_FREQ1(FRQD), /* Default frequency divider and osc offset */

//		DSP_DCPC0,
//		DSP_DCPC1(PRECHD | DISCHD), /* Default precharge, discharge timing */

//		DSP_CPADS0,
//		DSP_CPADS1(1), /* alternate (default) com pads config. Line interleave. */

//		DSP_VCOM0,
//		DSP_VCOM1(0x40), /* Set VCOM deselct voltage to high (1) */

		DSP_LPOW(1), /* display on */

		DSP_NOP, /* We use NOP command here as sentinel value */
		};
		const uint8_t *p = V;
		while (*p != DSP_NOP) dsp_cmd(*p++);
	}

	mdelay(100);

	/* Clear frame buffer and display */
	DSP_FILL(0);
	dsp_flush();

	if (0) { /* Game of life sequence */
		int i;
		dsp_snow(); dsp_snow();
		for (i = 0; i < 1000; i++) {
			dsp_life();
			dsp_flush();
			mdelay(40);
		}
	}

	if (0) { /* Pong sequence */
		int i;
		for (i = 0; i < 2000; i++) {
			dsp_pong(2, 2);
			dsp_flush();
			mdelay(10);
		}
	}

	return 0;
}

uint8_t dsp_FB[DSP_RESX * DSP_PAGES];

void dsp_flush(void)
{
	dsp_dta(sizeof(dsp_FB), dsp_FB);
}

void dsp_line(int x1, int y1, int x2, int y2, int c)
{
	int dx = abs(x2 - x1);
	int sx = y1 < y2 ? 1 : -1;
	int dy = abs(y2 - y1);
	int sy = y1 < y2 ? 1 : -1;
	int e  = (dx > dy ? dx : -dy) / 2;
	int eo;
	while(1) {
		DSP_PIXEL(x1, y1, c);
		if (x1 == x2 && y1 == y2) break;
		eo = e;
		if (eo > -dx) {
			e -= dy;
			x1 += sx;
		}
		if (eo < dy) {
			e += dx;
			y1 += sy;
		}
	}
}

void dsp_vline(int x, int y1, int y2, int c)
{
	int y;
	for (y = y1; y <= y2; y++)
		DSP_PIXEL(x, y, c);
}

void dsp_hline(int x1, int x2, int y, int c)
{
	int x;
	for (x = x1; x <= x2; x++)
		DSP_PIXEL(x, y, c);
}

void dsp_square(int x1, int y1, int x2, int y2, int c)
{
	int x;
	for (x = x1; x <= x2; x++) {
		dsp_vline(x, y1, y2, c);
	}
}

/* Extra stuff */

static uint16_t rand_16(void)
{
	static uint32_t rnd;
	uint16_t rv;
	const uint32_t a = 1103515245;
	const uint32_t c = 12345;
	rnd = rnd * a + c;
	rv = (rnd >> 16) ^ rnd;
	return rv;
}

/* Random pixels in framebuffer. Call repeatedly for snow effect */
void dsp_snow(void)
{
	int i;
	for (i = 0; i < sizeof(dsp_FB); i++ ) {
		dsp_FB[i] = rand_16() & 0xFF;
	}
	rand_16(); /* Offset to avoid slow column syndrome */
}

/* Wrapped game of life on framebuffer. Single step. */
void dsp_life(void)
{
	int x,y;
	uint8_t shadow[sizeof(dsp_FB)];
	for (y = 0; y < DSP_RESY; y++) {
		for (x = 0; x < DSP_RESX; x++) {
			int nb = 0;
			nb =    DSP_RPIXEL((x-1) % DSP_RESX, (y-1) % DSP_RESY) +
				DSP_RPIXEL((x+0) % DSP_RESX, (y-1) % DSP_RESY) +
				DSP_RPIXEL((x+1) % DSP_RESX, (y-1) % DSP_RESY) +
				DSP_RPIXEL((x-1) % DSP_RESX, (y+0) % DSP_RESY) +
				DSP_RPIXEL((x+1) % DSP_RESX, (y+0) % DSP_RESY) +
				DSP_RPIXEL((x-1) % DSP_RESX, (y+1) % DSP_RESY) +
				DSP_RPIXEL((x+0) % DSP_RESX, (y+1) % DSP_RESY) +
				DSP_RPIXEL((x+1) % DSP_RESX, (y+1) % DSP_RESY);
			if (DSP_RPIXEL(x, y)) {
				if ((nb < 2) || (nb > 3))
					shadow[DSP_BYTE((x),(y))] &= ~(1 << DSP_BIT((x),(y)));
			} else {
				if (nb == 3)
					shadow[DSP_BYTE((x),(y))] |= (1 << DSP_BIT((x),(y)));
			}
		}
	}
	memcpy(dsp_FB, shadow, sizeof(dsp_FB));
}

/* Game of pong, single step.
 * controllers: 2 AI, -1 up, 0 stay, 1 down
 * Returns -1 after miss, resets to center. */
int dsp_pong(int left, int right)
{
	int i;
	#define BS 3  // Ball size
	#define PS 16 // pallete hight
	static int yl, yr;
	static int bx = DSP_RESX / 2, bxx = 1;
	static int by = DSP_RESY / 2, byy = 1;
	int rval = 0;
	/* Clear */
	DSP_FILL(0);
	/* Draw court */
	for (i = 0; i < DSP_RESX; i++) {
		DSP_PIXEL(i, 0, i & 1);
		DSP_PIXEL(i, DSP_RESY - 1, i & 1);
	}
	for (i = 0; i < DSP_RESY; i++) DSP_PIXEL(DSP_RESX / 2, i, i & 1);
	/* Left bat */
	if (left == 2) {
		int d = yl - by - BS / 2 + PS /2;
		if (d > 0 && bxx < 0 && rand_16() % 3 == 0) yl -= 1;
		if (d < 0 && bxx < 0 && rand_16() % 3 == 0) yl += 1;
	} else {
		yl += left;
	}
	if (yl < 0) yl = 0;
	if (yl > DSP_RESY - PS/2) yl = DSP_RESY - PS/2;
	/* Right bat */
	if (right == 2) {
		int d = yr - by - BS / 2 + PS /2;
		if (d > 0 && bxx > 0 && rand_16() % 3 == 0) yr -= 1;
		if (d < 0 && bxx > 0 && rand_16() % 3 == 0) yr += 1;
	} else {
		yr += right;
	}
	if (yr < 0) yr = 0;
	if (yr > DSP_RESY - PS/2) yr = DSP_RESY - PS/2;
	/* Ball */
	bx += bxx;
	by += byy;
	if (bx == BS) {
		if (by < yl || by > yl + PS - BS) rval = 1;
		bxx = 1;
	}
	if (bx == DSP_RESX - 2*BS - 1) {
		if (by < yr || by > yr + PS - BS) rval = 1;
		bxx = -1;
	}
	if ((by == 0) || (by == DSP_RESY - 1 - BS)) {
		byy = -byy;
	}
	/* Draw stuff */
	dsp_square(bx, by, bx+BS, by+BS, 1); // Ball
	dsp_square(0, yl, BS, yl + PS, 1); // Left bat
	dsp_square(DSP_RESX - BS - 1, yr, DSP_RESX - 1, yr+PS, 1); // Right bat

	if (rval) {
		bx = DSP_RESX / 2;
		by = DSP_RESY / 2;
	}

	return rval;
}

