/*
This file may be included to implement CBM bitmap decompressor.
unsigned type must be at least 16 bits for bitmaps < 255x255, 32
bits otherwise.

CBM file format
====================

Lossless compressed bitmap format, microscopic decompress requirements
BW bitmap, 64kx64k max.

Max 3-5 byte overhead (header) for uncompressible data.

Header
-------

00 Flags (0xA0 | INVERT, COMPRESSED, LARGE, BF)

INVERT:     If set, bitmap is stored inverted, must be de-inverted on decode.
COMPRESSED: If set, bitmap is stored as RLE elements, packed pixels otherwise.
LARGE:      If set, header uses 2 bytes (Big Endian) per dimension to encode size.
BF:         IF set, skip and draw (but not raw) lengths are BF encoded (see below).

if (!large)
  01 Xsize (1-256)
  01 Ysize (1-256)
else
  01 02 Xsize (1-65536)
  03 04 Ysize (1-65536)


Packed pixels
---------------

Used for storing uncompressed bitmaps or raw RLE elements.

Pixels are packed 8 to a byte, MSB first.

In case the x size is not a multiple of 8, bytes run into
the next line without any alignment.

If total number of pixels is not divisible by 8, the
remaining (LSB) bits of the last byte are ignored.


BF encoding
-------------

This is a logarithmic encoding similar to a-law or u-law.

Block float encoding:

encoded  formula value
0-15   : *1      0-15
16-31  : *2-16   16-46
32-63  : *4-80   48-172
64-255 : *8-336  176-1704

Encoding is reversible and any number up to max+15 can be
expressed as the sum of 2 encoded numbers or less.

Less than 8 bit encoding sets unused MSB's to 0.

If the BF flag in the header is not set, BF encoding
reverts to 1:1 relationship globally.



RLE elements
--------------

Compressed bitmaps consist of RLE elements. An element
is a 2-byte entity with an optional data area following.

First byte is SKIP length, BF encoded in pixels. This
many pixels are drawn in background color (actual color
depends on INVERT).

Second byte contains a flag bit (RAW, 0x80) and a 7-bit
length specifier.

If RAW is set, the length specifier is in (pixels / 8),
never BF encoded. Length bytes of packed pixel data follow.
The length is never 0.

If RAW is not set, it's a DRAW element. The DRAW length
is BF encoded in pixels.


A raw block with 0 length can be used for future expansion,
it will never occur otherwise. Maybe repeating patterns?


Iwo Mergler
*/

#ifndef _CBM_DECODER_H
#define _CBM_DECODER_H

#define CBM_VERSION    0xA0
#define CBM_INVERT     0x08 /* bitmaped stored inverted */
#define CBM_COMPRESSED 0x04 /* bitmap stored compressed */
#define CBM_LARGE      0x02 /* Uses large (2b) size entries in header. */
#define CBM_BF         0x01 /* Uses BF number encoding. */

#define CBM_RAW        0x80
#define CBM_DLEN       0x7F
#define CBM_NONE       0x00

/* Returns bitmap x size, needs minimum of 3 bytes
 * in cbm buffer */
static inline unsigned CBM_xsize(const uint8_t *cbm)
{
	unsigned rval;
	if (cbm[0] & CBM_LARGE)
		rval = (unsigned)cbm[1] << 8 | cbm[2];
	else
		rval = cbm[1];
	return rval + 1;
}

/* Returns bitmap y size, needs minimum of 5 bytes
 * in cbm buffer */
static inline unsigned CBM_ysize(const uint8_t *cbm)
{
	unsigned rval;
	if (cbm[0] & CBM_LARGE)
		rval = (unsigned)cbm[3] << 8 | cbm[4];
	else
		rval = cbm[2];
	return rval + 1;
}

/* unpacks a byte bf into an unsigned 0-1704. If bf is not
 * set, a 1:1 translation is used. */
static inline unsigned CBM_bf_unpack(uint8_t v, int bf)
{
	unsigned x = v;
	if (!bf) return v;
	if (x < 16) return x;
	if (x < 32) return x * 2 - 16;
	if (x < 64) return x * 4 - 80;
	return x * 8 - 336;
}

static inline unsigned CBM_rw_unpack(uint8_t v)
{
	return v << 3; /* * 8 */
}

#if 0
#define CBMD(...) printf(__VA_ARGS__)
#else
#define CBMD(...) do{}while(0)
#endif

/* Decompresses bitmap, using provided pixel drawing function.
 * To apply an offset, make pixel() a wrapper that adds it. Same
 * for transparency maps. pixel() c value is boolean set/clear. */
static inline void CBM_decompress(const uint8_t *cbm, int bytes, void (*pixel)(unsigned x, unsigned y, int c))
{
	unsigned xs, ys, x, y;
	uint8_t flags;
	uint8_t v = 0, eh;
	unsigned n, m;

	/* To check header, assert(cbm[0] & 0xF0 == CBM_VERSION) */

	/* Flags */
	flags = cbm[0];

	/* Bitmap size */
	xs = CBM_xsize(cbm);
	ys = CBM_ysize(cbm);

	/* Skip header */
	cbm   += (flags & CBM_LARGE)?5:3;
	bytes -= (flags & CBM_LARGE)?5:3;

	/* Uncompressed? */
	if (!(flags & CBM_COMPRESSED)) {
		n = 0;
		for (y = 0; y < ys; y++) {
			for (x = 0; x < xs; x++) {
				if (!(n & 0x07)) v = *cbm++;
				pixel(x, y, !!(v & 0x80) ^ !!(flags & CBM_INVERT));
				v <<= 1; n++;
			}
		}
		return;
	}

	/* Compressed */
	x = y = 0;
	while (bytes > 0) { /* Minimum RLE entry is two bytes */
		bytes -= 2;
		CBMD("%d %02x %02x", bytes, cbm[0], cbm[1]);
		/* Skip */
		eh = *cbm++;
		n = CBM_bf_unpack(eh, flags & CBM_BF);
		CBMD(" S%d", n);
		while (n--) {
			pixel(x, y, 0 ^ !!(flags & CBM_INVERT));
			x++; if (x >= xs) {y++; x = 0;}
		}
		/* Draw */
		eh = *cbm++;
		if (eh & CBM_RAW) {
			/* Raw bitmap */
			m = CBM_rw_unpack(eh & CBM_DLEN);
			n = 0;
			CBMD(" R%d [", m);
			while(m--) {
				if (!(n & 0x7)) {
					v = *cbm++;
					bytes--;
					CBMD("%02x", v);
				}
				pixel(x, y, !!(v & 0x80) ^ !!(flags & CBM_INVERT));
				v <<= 1; n++;
				x++; if (x >= xs) {y++; x = 0;};
			}
			CBMD("]");
		} else {
			/* repeat draw */
			m = CBM_bf_unpack(eh, flags & CBM_BF);
			CBMD(" D%d", m);
			while (m--) {
				pixel(x, y, 1 ^ !!(flags & CBM_INVERT));
				x++; if (x >= xs) {y++; x = 0;};
			}
		}
		CBMD("\n");
	}

	/* Remaining background pixels */
	while (y < ys) {
		pixel(x, y, 0 ^ !!(flags & CBM_INVERT));
		x++; if (x >= xs) {y++; x = 0;}
	}
}

#endif /* !_CBM_DECODER_H */
