/*
 * Freescale i.MX28 Boot PMIC init
 *
 * Copyright (C) 2011 Marek Vasut <marek.vasut@gmail.com>
 * on behalf of DENX Software Engineering GmbH
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <config.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/iomux-mx28.h>
#include <asm/arch/sys_proto.h>
#include <asm/gpio.h>

#include "mx28_init.h"

#ifndef CONFIG_SPL_BUILD
#define early_delay udelay
#endif

#define DELAY_BEFORE_RESTORE_SETTINGS_US 25
#define RAMP_UP_CURRENT_LIMIT_TIME_US 100
#define AC_CHARGE_4P2_ILIMIT 0x3f

#ifdef CONFIG_POWER_STATE_MACHINE
#define BM_REQUEST_PERSISTENT 0xFFFF
#define LOW_BATT_REQUEST_PERSISTENT 0x1111
#define SHUTDOWN_REQUEST_PERSISTENT 0x2222
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V64 0x1F
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V6 0x1E
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V56 0x1D
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V52 0x1C
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V48 0x1B
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V44 0x1A
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V4 0x19
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V36 0x18
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V32 0x17
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V28 0x16
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V24 0x15
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V2 0x14
#define POWER_BATTMONITOR_BRWNOUT_LVL_3V 0xF
/* time unit: WAIT_FOR_CORRECT_BATT_MEASUREMENT_US */
#define WAIT_FOR_CORRECT_BATT_MEASUREMENT 20
#define WAIT_FOR_CORRECT_BATT_MEASUREMENT_US 100000
#define VALID_BATT_VOLT_MV 1000
#define BATT_VOLT_OK_TO_BOOT 3750
#define DELAY_TO_CHECK_BATTERY_US 20000
#define BATT_VOLT_THRESH_REENABLE_CHARGER 4000
#define DELAY_FOR_STATE_MACHINE_CYCLE_US 1000000
#define WATCHDOG_TIMER_STATE_MACHINE_MS 5000
#endif

static void mx28_power_clock2xtal(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

	/* Set XTAL as CPU reference clock */
	writel(CLKCTRL_CLKSEQ_BYPASS_CPU,
		&clkctrl_regs->hw_clkctrl_clkseq_set);
}

static void mx28_power_clock2pll(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

	setbits_le32(&clkctrl_regs->hw_clkctrl_pll0ctrl0,
			CLKCTRL_PLL0CTRL0_POWER);
	early_delay(100);
	setbits_le32(&clkctrl_regs->hw_clkctrl_clkseq,
			CLKCTRL_CLKSEQ_BYPASS_CPU);
}

static void mx28_pll0_off(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
	clrbits_le32(&clkctrl_regs->hw_clkctrl_pll0ctrl0,
			CLKCTRL_PLL0CTRL0_POWER);
	early_delay(100);
}

static void mx28_power_clear_auto_restart(void)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;

	/* Take the RTC out of reset */
	writel(RTC_CTRL_SFTRST, &rtc_regs->hw_rtc_ctrl_clr);
	while (readl(&rtc_regs->hw_rtc_ctrl) & RTC_CTRL_SFTRST)
		;

	/* Make sure the RTC's clock is not gated */
	writel(RTC_CTRL_CLKGATE, &rtc_regs->hw_rtc_ctrl_clr);
	while (readl(&rtc_regs->hw_rtc_ctrl) & RTC_CTRL_CLKGATE)
		;

	/*
	 * Due to the hardware design bug of mx28 EVK-A
	 * we need to set the AUTO_RESTART bit.
	 */
	if (readl(&rtc_regs->hw_rtc_persistent0) & RTC_PERSISTENT0_AUTO_RESTART)
		return;

	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK)
		;

	setbits_le32(&rtc_regs->hw_rtc_persistent0,
			RTC_PERSISTENT0_AUTO_RESTART);
	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_set);
	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_clr);
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK)
		;
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_STALE_REGS_MASK)
		;
}

static void mx28_power_set_linreg(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/* Set linear regulator 25mV below switching converter */
	clrsetbits_le32(&power_regs->hw_power_vdddctrl,
			POWER_VDDDCTRL_LINREG_OFFSET_MASK,
			POWER_VDDDCTRL_LINREG_OFFSET_1STEPS_BELOW);

	clrsetbits_le32(&power_regs->hw_power_vddactrl,
			POWER_VDDACTRL_LINREG_OFFSET_MASK,
			POWER_VDDACTRL_LINREG_OFFSET_1STEPS_BELOW);

	clrsetbits_le32(&power_regs->hw_power_vddioctrl,
			POWER_VDDIOCTRL_LINREG_OFFSET_MASK,
			POWER_VDDIOCTRL_LINREG_OFFSET_1STEPS_BELOW);
}

#if defined(CONFIG_POWER_STATE_MACHINE) || (!defined(CONFIG_NO_VDD5V) && !defined(CONFIG_UNIFIED_POWER_INIT_PATH))
static int mx28_get_batt_volt(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t volt = readl(&power_regs->hw_power_battmonitor);
	volt &= POWER_BATTMONITOR_BATT_VAL_MASK;
	volt >>= POWER_BATTMONITOR_BATT_VAL_OFFSET;
	volt *= 8;
	return volt;
}
#endif

#if !defined(CONFIG_NO_VDD5V) && !defined(CONFIG_UNIFIED_POWER_INIT_PATH)
static int mx28_is_batt_ready(void)
{
	return (mx28_get_batt_volt() >= 3600);
}

static int mx28_is_batt_good(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t volt = mx28_get_batt_volt();

	if ((volt >= 2400) && (volt <= 4300))
		return 1;

	clrsetbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK,
		0x3 << POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET);
	writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
		&power_regs->hw_power_5vctrl_clr);

	clrsetbits_le32(&power_regs->hw_power_charge,
		POWER_CHARGE_STOP_ILIMIT_MASK | POWER_CHARGE_BATTCHRG_I_MASK,
		POWER_CHARGE_STOP_ILIMIT_10MA | 0x3);

	writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_clr);
	writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
		&power_regs->hw_power_5vctrl_clr);

	early_delay(500000);

	volt = mx28_get_batt_volt();

	if (volt >= 3500)
		return 0;

	if (volt >= 2400)
		return 1;

	writel(POWER_CHARGE_STOP_ILIMIT_MASK | POWER_CHARGE_BATTCHRG_I_MASK,
		&power_regs->hw_power_charge_clr);
	writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_set);

	return 0;
}
#endif
static void mx28_power_setup_5v_detect(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/* Start 5V detection */
	clrsetbits_le32(&power_regs->hw_power_5vctrl,
			POWER_5VCTRL_VBUSVALID_TRSH_MASK,
			POWER_5VCTRL_VBUSVALID_TRSH_4V4 |
			POWER_5VCTRL_PWRUP_VBUS_CMPS);
}

static void mx28_src_power_init(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/* Improve efficieny and reduce transient ripple */
	writel(POWER_LOOPCTRL_TOGGLE_DIF | POWER_LOOPCTRL_EN_CM_HYST |
		POWER_LOOPCTRL_EN_DF_HYST, &power_regs->hw_power_loopctrl_set);

	clrsetbits_le32(&power_regs->hw_power_dclimits,
			POWER_DCLIMITS_POSLIMIT_BUCK_MASK,
			0x30 << POWER_DCLIMITS_POSLIMIT_BUCK_OFFSET);

	setbits_le32(&power_regs->hw_power_battmonitor,
			POWER_BATTMONITOR_EN_BATADJ);

	/* Increase the RCSCALE level for quick DCDC response to dynamic load */
	clrsetbits_le32(&power_regs->hw_power_loopctrl,
			POWER_LOOPCTRL_EN_RCSCALE_MASK,
			POWER_LOOPCTRL_RCSCALE_THRESH |
			POWER_LOOPCTRL_EN_RCSCALE_8X);

	clrsetbits_le32(&power_regs->hw_power_minpwr,
			POWER_MINPWR_HALFFETS, POWER_MINPWR_DOUBLE_FETS);

#ifndef CONFIG_UNIFIED_POWER_INIT_PATH
	/* 5V to battery handoff ... FIXME */
	setbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_DCDC_XFER);
	early_delay(30);
	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_DCDC_XFER);
#endif
}

#if !defined(CONFIG_NO_VDD5V) && !defined(CONFIG_UNIFIED_POWER_INIT_PATH)
static void mx28_power_init_4p2_params(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/* Setup 4P2 parameters */
	clrsetbits_le32(&power_regs->hw_power_dcdc4p2,
		POWER_DCDC4P2_CMPTRIP_MASK | POWER_DCDC4P2_TRG_MASK,
		POWER_DCDC4P2_TRG_4V2 | (31 << POWER_DCDC4P2_CMPTRIP_OFFSET));

	clrsetbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_HEADROOM_ADJ_MASK,
		0x4 << POWER_5VCTRL_HEADROOM_ADJ_OFFSET);

	clrsetbits_le32(&power_regs->hw_power_dcdc4p2,
		POWER_DCDC4P2_DROPOUT_CTRL_MASK,
		POWER_DCDC4P2_DROPOUT_CTRL_100MV |
		POWER_DCDC4P2_DROPOUT_CTRL_SRC_SEL);

	clrsetbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK,
		0x3f << POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET);
}

static void mx28_enable_4p2_dcdc_input(int xfer)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t tmp, vbus_thresh, vbus_5vdetect, pwd_bo;
	uint32_t prev_5v_brnout, prev_5v_droop;

	prev_5v_brnout = readl(&power_regs->hw_power_5vctrl) &
				POWER_5VCTRL_PWDN_5VBRNOUT;
	prev_5v_droop = readl(&power_regs->hw_power_ctrl) &
				POWER_CTRL_ENIRQ_VDD5V_DROOP;

	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_PWDN_5VBRNOUT);
	writel(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD_OFF,
		&power_regs->hw_power_reset);

	clrbits_le32(&power_regs->hw_power_ctrl, POWER_CTRL_ENIRQ_VDD5V_DROOP);

	if (xfer && (readl(&power_regs->hw_power_5vctrl) &
			POWER_5VCTRL_ENABLE_DCDC)) {
		return;
	}

	/*
	 * Recording orignal values that will be modified temporarlily
	 * to handle a chip bug. See chip errata for CQ ENGR00115837
	 */
	tmp = readl(&power_regs->hw_power_5vctrl);
	vbus_thresh = tmp & POWER_5VCTRL_VBUSVALID_TRSH_MASK;
	vbus_5vdetect = tmp & POWER_5VCTRL_VBUSVALID_5VDETECT;

	pwd_bo = readl(&power_regs->hw_power_minpwr) & POWER_MINPWR_PWD_BO;

	/*
	 * Disable mechanisms that get erroneously tripped by when setting
	 * the DCDC4P2 EN_DCDC
	 */
	clrbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_VBUSVALID_5VDETECT |
		POWER_5VCTRL_VBUSVALID_TRSH_MASK);

	writel(POWER_MINPWR_PWD_BO, &power_regs->hw_power_minpwr_set);

	if (xfer) {
		setbits_le32(&power_regs->hw_power_5vctrl,
				POWER_5VCTRL_DCDC_XFER);
		early_delay(20);
		clrbits_le32(&power_regs->hw_power_5vctrl,
				POWER_5VCTRL_DCDC_XFER);

		setbits_le32(&power_regs->hw_power_5vctrl,
				POWER_5VCTRL_ENABLE_DCDC);
	} else {
		setbits_le32(&power_regs->hw_power_dcdc4p2,
				POWER_DCDC4P2_ENABLE_DCDC);
	}

	early_delay(25);

	clrsetbits_le32(&power_regs->hw_power_5vctrl,
			POWER_5VCTRL_VBUSVALID_TRSH_MASK, vbus_thresh);

	if (vbus_5vdetect)
		writel(vbus_5vdetect, &power_regs->hw_power_5vctrl_set);

	if (!pwd_bo)
		clrbits_le32(&power_regs->hw_power_minpwr, POWER_MINPWR_PWD_BO);

	while (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VBUS_VALID_IRQ)
		writel(POWER_CTRL_VBUS_VALID_IRQ,
			&power_regs->hw_power_ctrl_clr);

	if (prev_5v_brnout) {
		writel(POWER_5VCTRL_PWDN_5VBRNOUT,
			&power_regs->hw_power_5vctrl_set);
		writel(POWER_RESET_UNLOCK_KEY,
			&power_regs->hw_power_reset);
	} else {
		writel(POWER_5VCTRL_PWDN_5VBRNOUT,
			&power_regs->hw_power_5vctrl_clr);
		writel(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD_OFF,
			&power_regs->hw_power_reset);
	}

	while (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VDD5V_DROOP_IRQ)
		writel(POWER_CTRL_VDD5V_DROOP_IRQ,
			&power_regs->hw_power_ctrl_clr);

	if (prev_5v_droop)
		clrbits_le32(&power_regs->hw_power_ctrl,
				POWER_CTRL_ENIRQ_VDD5V_DROOP);
	else
		setbits_le32(&power_regs->hw_power_ctrl,
				POWER_CTRL_ENIRQ_VDD5V_DROOP);
}

static void mx28_power_init_4p2_regulator(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t tmp, tmp2;

	setbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_ENABLE_4P2);

	writel(POWER_CHARGE_ENABLE_LOAD, &power_regs->hw_power_charge_set);

	writel(POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK,
		&power_regs->hw_power_5vctrl_clr);
	clrbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_TRG_MASK);

	/* Power up the 4p2 rail and logic/control */
	writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
		&power_regs->hw_power_5vctrl_clr);

	/*
	 * Start charging up the 4p2 capacitor. We ramp of this charge
	 * gradually to avoid large inrush current from the 5V cable which can
	 * cause transients/problems
	 */
	mx28_enable_4p2_dcdc_input(0);

	if (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VBUS_VALID_IRQ) {
		/*
		 * If we arrived here, we were unable to recover from mx23 chip
		 * errata 5837. 4P2 is disabled and sufficient battery power is
		 * not present. Exiting to not enable DCDC power during 5V
		 * connected state.
		 */
		clrbits_le32(&power_regs->hw_power_dcdc4p2,
			POWER_DCDC4P2_ENABLE_DCDC);
		writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
			&power_regs->hw_power_5vctrl_set);
		hang();
	}

	/*
	 * Here we set the 4p2 brownout level to something very close to 4.2V.
	 * We then check the brownout status. If the brownout status is false,
	 * the voltage is already close to the target voltage of 4.2V so we
	 * can go ahead and set the 4P2 current limit to our max target limit.
	 * If the brownout status is true, we need to ramp us the current limit
	 * so that we don't cause large inrush current issues. We step up the
	 * current limit until the brownout status is false or until we've
	 * reached our maximum defined 4p2 current limit.
	 */
	clrsetbits_le32(&power_regs->hw_power_dcdc4p2,
			POWER_DCDC4P2_BO_MASK,
			22 << POWER_DCDC4P2_BO_OFFSET);	/* 4.15V */

	if (!(readl(&power_regs->hw_power_sts) & POWER_STS_DCDC_4P2_BO)) {
		setbits_le32(&power_regs->hw_power_5vctrl,
			0x3f << POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET);
	} else {
		tmp = (readl(&power_regs->hw_power_5vctrl) &
			POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK) >>
			POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET;
		while (tmp < 0x3f) {
			if (!(readl(&power_regs->hw_power_sts) &
					POWER_STS_DCDC_4P2_BO)) {
				tmp = readl(&power_regs->hw_power_5vctrl);
				tmp |= POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK;
				early_delay(100);
				writel(tmp, &power_regs->hw_power_5vctrl);
				break;
			} else {
				tmp++;
				tmp2 = readl(&power_regs->hw_power_5vctrl);
				tmp2 &= ~POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK;
				tmp2 |= tmp <<
					POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET;
				writel(tmp2, &power_regs->hw_power_5vctrl);
				early_delay(100);
			}
		}
	}

	clrbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_BO_MASK);
	writel(POWER_CTRL_DCDC4P2_BO_IRQ, &power_regs->hw_power_ctrl_clr);
}

static void mx28_power_init_dcdc_4p2_source(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	if (!(readl(&power_regs->hw_power_dcdc4p2) &
		POWER_DCDC4P2_ENABLE_DCDC)) {
		hang();
	}

	mx28_enable_4p2_dcdc_input(1);

	if (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VBUS_VALID_IRQ) {
		clrbits_le32(&power_regs->hw_power_dcdc4p2,
			POWER_DCDC4P2_ENABLE_DCDC);
		writel(POWER_5VCTRL_ENABLE_DCDC,
			&power_regs->hw_power_5vctrl_clr);
		writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
			&power_regs->hw_power_5vctrl_set);
	}
}

static void mx28_power_enable_4p2(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t vdddctrl, vddactrl, vddioctrl;
	uint32_t tmp;

	vdddctrl = readl(&power_regs->hw_power_vdddctrl);
	vddactrl = readl(&power_regs->hw_power_vddactrl);
	vddioctrl = readl(&power_regs->hw_power_vddioctrl);

	setbits_le32(&power_regs->hw_power_vdddctrl,
		POWER_VDDDCTRL_DISABLE_FET | POWER_VDDDCTRL_ENABLE_LINREG |
		POWER_VDDDCTRL_PWDN_BRNOUT);

	setbits_le32(&power_regs->hw_power_vddactrl,
		POWER_VDDACTRL_DISABLE_FET | POWER_VDDACTRL_ENABLE_LINREG |
		POWER_VDDACTRL_PWDN_BRNOUT);

	setbits_le32(&power_regs->hw_power_vddioctrl,
		POWER_VDDIOCTRL_DISABLE_FET | POWER_VDDIOCTRL_PWDN_BRNOUT);

	mx28_power_init_4p2_params();
	mx28_power_init_4p2_regulator();

	/* Shutdown battery (none present) */
	if (!mx28_is_batt_ready()) {
		clrbits_le32(&power_regs->hw_power_dcdc4p2,
				POWER_DCDC4P2_BO_MASK);
		writel(POWER_CTRL_DCDC4P2_BO_IRQ,
				&power_regs->hw_power_ctrl_clr);
		writel(POWER_CTRL_ENIRQ_DCDC4P2_BO,
				&power_regs->hw_power_ctrl_clr);
	}

	mx28_power_init_dcdc_4p2_source();

	writel(vdddctrl, &power_regs->hw_power_vdddctrl);
	early_delay(20);
	writel(vddactrl, &power_regs->hw_power_vddactrl);
	early_delay(20);
	writel(vddioctrl, &power_regs->hw_power_vddioctrl);

	/*
	 * Check if FET is enabled on either powerout and if so,
	 * disable load.
	 */
	tmp = 0;
	tmp |= !(readl(&power_regs->hw_power_vdddctrl) &
			POWER_VDDDCTRL_DISABLE_FET);
	tmp |= !(readl(&power_regs->hw_power_vddactrl) &
			POWER_VDDACTRL_DISABLE_FET);
	tmp |= !(readl(&power_regs->hw_power_vddioctrl) &
			POWER_VDDIOCTRL_DISABLE_FET);
	if (tmp)
		writel(POWER_CHARGE_ENABLE_LOAD,
			&power_regs->hw_power_charge_clr);
}

static void mx28_boot_valid_5v(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/*
	 * Use VBUSVALID level instead of VDD5V_GT_VDDIO level to trigger a 5V
	 * disconnect event. FIXME
	 */
	writel(POWER_5VCTRL_VBUSVALID_5VDETECT,
		&power_regs->hw_power_5vctrl_set);

	/* Configure polarity to check for 5V disconnection. */
	writel(POWER_CTRL_POLARITY_VBUSVALID |
		POWER_CTRL_POLARITY_VDD5V_GT_VDDIO,
		&power_regs->hw_power_ctrl_clr);

	writel(POWER_CTRL_VBUS_VALID_IRQ | POWER_CTRL_VDD5V_GT_VDDIO_IRQ,
		&power_regs->hw_power_ctrl_clr);

	mx28_power_enable_4p2();
}
#endif

static void mx28_powerdown(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	writel(POWER_RESET_UNLOCK_KEY, &power_regs->hw_power_reset);
	writel(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD_OFF,
		&power_regs->hw_power_reset);
}

#ifndef CONFIG_UNIFIED_POWER_INIT_PATH
static void mx28_batt_boot(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_PWDN_5VBRNOUT);
	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_ENABLE_DCDC);

	clrbits_le32(&power_regs->hw_power_dcdc4p2,
			POWER_DCDC4P2_ENABLE_DCDC | POWER_DCDC4P2_ENABLE_4P2);
	writel(POWER_CHARGE_ENABLE_LOAD, &power_regs->hw_power_charge_clr);

	/* 5V to battery handoff. */
	setbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_DCDC_XFER);
	early_delay(30);
	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_DCDC_XFER);

	writel(POWER_CTRL_ENIRQ_DCDC4P2_BO, &power_regs->hw_power_ctrl_clr);

	clrsetbits_le32(&power_regs->hw_power_minpwr,
			POWER_MINPWR_HALFFETS, POWER_MINPWR_DOUBLE_FETS);

	mx28_power_set_linreg();

	clrbits_le32(&power_regs->hw_power_vdddctrl,
		POWER_VDDDCTRL_DISABLE_FET | POWER_VDDDCTRL_ENABLE_LINREG);

	clrbits_le32(&power_regs->hw_power_vddactrl,
		POWER_VDDACTRL_DISABLE_FET | POWER_VDDACTRL_ENABLE_LINREG);

	clrbits_le32(&power_regs->hw_power_vddioctrl,
		POWER_VDDIOCTRL_DISABLE_FET);

	setbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_PWD_CHARGE_4P2_MASK);

	setbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_ENABLE_DCDC);

	clrsetbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK,
		0x8 << POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET);
}
#endif

#if !defined(CONFIG_NO_VDD5V) && !defined(CONFIG_UNIFIED_POWER_INIT_PATH)
static void mx28_handle_5v_conflict(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t tmp;

	setbits_le32(&power_regs->hw_power_vddioctrl,
			POWER_VDDIOCTRL_BO_OFFSET_MASK);

	for (;;) {
		tmp = readl(&power_regs->hw_power_sts);

		if (tmp & POWER_STS_VDDIO_BO) {
			mx28_powerdown();
			break;
		}

		if (tmp & POWER_STS_VDD5V_GT_VDDIO) {
			mx28_boot_valid_5v();
			break;
		} else {
			mx28_powerdown();
			break;
		}

		if (tmp & POWER_STS_PSWITCH_MASK) {
			mx28_batt_boot();
			break;
		}
	}
}

static void mx28_5v_boot(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/*
	 * NOTE: In original IMX-Bootlets, this also checks for VBUSVALID,
	 * but their implementation always returns 1 so we omit it here.
	 */
	if (readl(&power_regs->hw_power_sts) & POWER_STS_VDD5V_GT_VDDIO) {
		mx28_boot_valid_5v();
		return;
	}

	early_delay(1000);
	if (readl(&power_regs->hw_power_sts) & POWER_STS_VDD5V_GT_VDDIO) {
		mx28_boot_valid_5v();
		return;
	}

	mx28_handle_5v_conflict();
}
#endif
#ifndef CONFIG_UNIFIED_POWER_INIT_PATH
static void mx28_init_batt_bo(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	/* Brownout at 3V */
	clrsetbits_le32(&power_regs->hw_power_battmonitor,
		POWER_BATTMONITOR_BRWNOUT_LVL_MASK,
		15 << POWER_BATTMONITOR_BRWNOUT_LVL_OFFSET);

	writel(POWER_CTRL_BATT_BO_IRQ, &power_regs->hw_power_ctrl_clr);
	writel(POWER_CTRL_ENIRQ_BATT_BO, &power_regs->hw_power_ctrl_clr);
}
#endif
static void mx28_switch_vddd_to_dcdc_source(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	clrsetbits_le32(&power_regs->hw_power_vdddctrl,
		POWER_VDDDCTRL_LINREG_OFFSET_MASK,
		POWER_VDDDCTRL_LINREG_OFFSET_1STEPS_BELOW);

	clrbits_le32(&power_regs->hw_power_vdddctrl,
		POWER_VDDDCTRL_DISABLE_FET | POWER_VDDDCTRL_ENABLE_LINREG |
		POWER_VDDDCTRL_DISABLE_STEPPING);
}

#ifdef CONFIG_POWER_STATE_MACHINE
#include <asm/arch/regs-usbphy.h>
#define LRADC_AC_IN_MON 5
#define LRADC_5V_THRESHOLD 2200
#define USB_HOST_CHARGE_ILIMIT_100MA 0x8
#define CHARGER_ILIMIT 0x3f
/*
 * Enable detection of USB D+ and D- plugged in
 */
static void mx28_enable_usb_plugin_detect(void)
{
	struct mx28_usbphy_regs *usbphy_regs = (struct mx28_usbphy_regs *)MXS_USBPHY0_BASE;
	/*
	 * HW_USBPHY_CTRL[ENDEVPLUGINDETECT]: For device mode, enables 200-KOhm pullups for
	 *  detecting connectivity to the host.
	 */
	writel(USBPHY_CTRL_ENDEVPLUGINDETECT, &usbphy_regs->hw_usbphy_ctrl_set);
}
/*
 * disable detection of USB D+ and D- plugged in
 */
void mx28_disable_usb_plugin_detect(void)
{
	struct mx28_usbphy_regs *usbphy_regs = (struct mx28_usbphy_regs *)MXS_USBPHY0_BASE;
	/*
	 * HW_USBPHY_CTRL[ENDEVPLUGINDETECT]: For device mode, enables 200-KOhm pullups for
	 *  detecting connectivity to the host.
	 */
	writel(USBPHY_CTRL_ENDEVPLUGINDETECT, &usbphy_regs->hw_usbphy_ctrl_clr);
}
/*
 * Check whether USB D+ and D- detected
 * mx28_enable_usb_plugin_detect must be already called
 * to enable detection
 *
 * Returns:
 * 	1:	detected
 * 	0: 	not detected
 */
static int mx28_is_usb_data_signal_detected(void)
{
	struct mx28_usbphy_regs *usbphy_regs = (struct mx28_usbphy_regs *)MXS_USBPHY0_BASE;
	/* HW_USBPHY_STATUS[DEVPLUGIN_STATUS] == 1 indicates that the device has been connected
	 *  on the USB_DP and USB_DM lines. */
	return ((readl(&usbphy_regs->hw_usbphy_status) & USBPHY_STATUS_DEVPLUGIN_STATUS) != 0);
}
#define LRADC0	0
#define LRADC1	1
#define LRADC2	2
#define LRADC3	3
#define LRADC4	4
#define LRADC5	5
#define LRADC6	6
/* Measure a LRADC channel
 * Parameters:
 * 	lradc_num:	LRADC input number, must be 0-6
 * 	div2:		1: Enable DIV2; 0: Disable DIV2
 *
 * Returns: measured voltage in mV. -1: error
 */
static int mx28_measure_lradc(int lradc_num, int div2)
{
#define BP_LRADC_CTRL2_DIVIDE_BY_TWO	24
#define BN_LRADC_CTRL4_MUX_SELECTOR	4
#define NUM_LRADC_READING	3
#define BM_LRADC_CH_VALUE 0x0003FFFF
#define WAIT_CONVERSION_US	10
#define MUX_TO_CHANNEL	2
	int  value, lradc_irq_mask,channel, sum = 0;
	int  out_value;
	int i;
	struct mx28_lradc_regs *lradc_regs = (struct mx28_lradc_regs *)MXS_LRADC_BASE;
	if (lradc_num<LRADC0 || lradc_num>LRADC6){
		return -1;
	}
	channel = MUX_TO_CHANNEL;
	lradc_irq_mask = 1 << channel;
	/* mux analog input lradc 1 for conversion on LRADC channel . */
	writel((0xF << (BN_LRADC_CTRL4_MUX_SELECTOR * channel)), &lradc_regs->hw_lradc_ctrl4_clr);
	writel((lradc_num << (BN_LRADC_CTRL4_MUX_SELECTOR * channel)), &lradc_regs->hw_lradc_ctrl4_set);
	if (div2){
		/* enable div2*/
		writel( (1 << channel) << BP_LRADC_CTRL2_DIVIDE_BY_TWO, &lradc_regs->hw_lradc_ctrl2_set);
	}
	else{
		/* disable div2*/
		writel( (1 << channel) << BP_LRADC_CTRL2_DIVIDE_BY_TWO, &lradc_regs->hw_lradc_ctrl2_clr);
	}
	for (i = 0; i < NUM_LRADC_READING; i++) {
		/* Clear the interrupt flag */
		writel(lradc_irq_mask, &lradc_regs->hw_lradc_ctrl1_clr);
		/* schedule */
		writel(1 << channel, &lradc_regs->hw_lradc_ctrl0_set);
	    /* Wait for conversion complete*/
	    while (!(readl(&lradc_regs->hw_lradc_ctrl1) & lradc_irq_mask)){
	    	early_delay(WAIT_CONVERSION_US);
	    }
	    /* Clear the interrupt flag again */
	    writel(lradc_irq_mask, &lradc_regs->hw_lradc_ctrl1_clr);
	    /* read adc value and clr lradc */
	    value = readl(&lradc_regs->hw_lradc_ch2) & BM_LRADC_CH_VALUE;
	    writel(BM_LRADC_CH_VALUE, &lradc_regs->hw_lradc_ch2_clr);
	    sum += value;
	}
	if (div2){
		/* disable div2*/
		writel( (1 << channel) << BP_LRADC_CTRL2_DIVIDE_BY_TWO, &lradc_regs->hw_lradc_ctrl2_clr);
	}
	/* Take the voltage average.  */
	value = sum/NUM_LRADC_READING;
	/* convert digital value to the real voltage   */
	/* input voltage = (ADC value) * 1.85/(2^12-1)  */
	out_value = value * 1850/4095; /* unit: mV */
	if (div2){
		out_value *= 2;
	}
	return out_value;
}
/*
 * turn on/off temperature current source
 *
 * Parameters:
 * 	analog_mux_input:	LRADC input, only LRADC0 and LRADC6 include external temperature sensor
 * 	on:					1: ON; 0: OFF
 */
static void mx28_turn_temp_current_source(int analog_mux_input, int on)
{
#define TIME_TEMP_SOURCE_CURRENT_RAMP_UP_US	1000
	struct mx28_lradc_regs *lradc_regs = (struct mx28_lradc_regs *)MXS_LRADC_BASE;
	/* only LRADC0 and LRADC6 include external temperature sensor */
	if (analog_mux_input!=LRADC0 && analog_mux_input!=LRADC6){
		return;
	}
	if (on)
	{
		/* setup source current 20uA */
		if (analog_mux_input == LRADC0){
			writel(LRADC_CTRL2_TEMP_ISRC0_20, &lradc_regs->hw_lradc_ctrl2_set);
			writel(LRADC_CTRL2_TEMP_SENSOR_IENABLE0, &lradc_regs->hw_lradc_ctrl2_set);
		}
		else{
			writel(LRADC_CTRL2_TEMP_ISRC1_20, &lradc_regs->hw_lradc_ctrl2_set);
			writel(LRADC_CTRL2_TEMP_SENSOR_IENABLE1, &lradc_regs->hw_lradc_ctrl2_set);
		}
		/* wait while the current ramps up.  */
		early_delay(TIME_TEMP_SOURCE_CURRENT_RAMP_UP_US);
	}
	else{
		/* turn off source current */
		if (analog_mux_input == LRADC0){
			writel(LRADC_CTRL2_TEMP_ISRC0_MASK, &lradc_regs->hw_lradc_ctrl2_clr);
			writel(LRADC_CTRL2_TEMP_SENSOR_IENABLE0, &lradc_regs->hw_lradc_ctrl2_clr);
		}
		else{
			writel(LRADC_CTRL2_TEMP_ISRC1_MASK, &lradc_regs->hw_lradc_ctrl2_clr);
			writel(LRADC_CTRL2_TEMP_SENSOR_IENABLE1, &lradc_regs->hw_lradc_ctrl2_clr);
		}
	}
}
/* wait for correct battery voltage measurement */
static void mx28_wait_for_correct_batt_measurement(void)
{
	int wait_time; /* time unit: WAIT_FOR_CORRECT_BATT_MEASUREMENT_US */
	for (wait_time = 0; wait_time < WAIT_FOR_CORRECT_BATT_MEASUREMENT; wait_time++){
		if (mx28_get_batt_volt() < VALID_BATT_VOLT_MV) {
			udelay(WAIT_FOR_CORRECT_BATT_MEASUREMENT_US);
		} else
			break;
	}
}
/*
 * check whether battery is detected
 *
 * Returns:
 * 	0: not detected
 * 	1: detected
 */
#define DETECTING_BATTERY_BY_BATT_THERMISTOR_VOLT
static int mx28_is_batt_detected(void)
{
#ifdef DETECTING_BATTERY_BY_BATT_THERMISTOR_VOLT
#ifdef HW_EVT3
/* With the DIVIDE_BY_TWO option set, the maximum input voltage is VDDIO - 50mv */
#define MAX_LRADC_INPUT_VOLT 3250
#else
/* (100KOhm is parallel to the thermistor --> Vmax >= I*100KOhm) */
#define MAX_LRADC_INPUT_VOLT 2000
#endif
	int batt_temp_volt;
	/*
	 * If battery is not present, voltage at battery thermistor monitoring line is at maximum value i.e ~2000mV on DVT, 3250mV on NTC-20 EVT3.
	 * Requirement: ADC for measuring battery thermistor monitoring line must be already setup.
	 */
	/* turn on current */
	mx28_turn_temp_current_source(LRADC0,1);
	udelay(DELAY_TO_CHECK_BATTERY_US);

	batt_temp_volt = mx28_measure_lradc(LRADC0,1);
	/* turn off current */
	mx28_turn_temp_current_source(LRADC0,0);

	if (batt_temp_volt>=MAX_LRADC_INPUT_VOLT){
		return 0;
	}
	else{
		return 1;
	}
#else
	int result;
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;

	/* set battery bo threshold */
	writel(  (readl(&power_regs->hw_power_battmonitor) & (~POWER_BATTMONITOR_BRWNOUT_LVL_MASK))
			| (POWER_BATTMONITOR_BRWNOUT_LVL_3V52 << POWER_BATTMONITOR_BRWNOUT_LVL_OFFSET), &power_regs->hw_power_battmonitor);

	/*
	1. Enable the automatic battery voltage update for the DC-DC control logic. This is performed to
	achieve an accurate battery voltage measurement.
	*/
	/* should already done */
	/*
	2. Configure the LinReg for the 5 V power source. This is the default configuration when the 5 V
	supply powers ON the device.
	*/

	/*
	3. Disable the battery-brownout power down and enable the battery-brownout comparator:
	— Set HW_POWER_BATTMONITOR[PWD_BATTBRNOUT] = 0
	— Set HW_POWER_BATTMONITOR[BRWNOUT_PWD] = 0
	*/
	writel(  (readl(&power_regs->hw_power_battmonitor) & (~POWER_BATTMONITOR_PWDN_BATTBRNOUT)) & (~POWER_BATTMONITOR_BRWNOUT_PWD),
			&power_regs->hw_power_battmonitor);

	/*
	4. Enable the fast-settling bit for the battery detection:
	HW_POWER_REFCTRL[FASTSETTLING] = 1
	*/
	writel(  POWER_REFCTRL_FASTSETTLING, &power_regs->hw_power_refctrl_set);

	/*
	5. Read the HW_POWER_STS[BATT_BO] bit:
	— If the bit is 1, battery brownout has occurred and the battery is not available
	— If the bit is 0, a battery is attached to the system
	*/
	if ( (readl(  &power_regs->hw_power_sts) & POWER_STS_BATT_BO ) != 0){
		result = 0;
	}
	else{
		result = 1;
	}

	/*
	6. Reset the fast-settling bit to complete the battery detection:
	HW_POWER_REFCTRL_CLR[FASTSETTLING] = 1
	*/
	writel(  POWER_REFCTRL_FASTSETTLING, &power_regs->hw_power_refctrl_clr);

	mx28_wait_for_correct_batt_measurement();

	return result;
#endif
}
/* check whether battery is ok to boot */
static int mx28_is_batt_ok_to_boot(void)
{
	return (mx28_get_batt_volt() >= BATT_VOLT_OK_TO_BOOT);
}
/*
 * turn LED on/off
 * Parameters:
 * 	on:	1:ON;0:OFF
 */
void mx28_turn_on_led(int on){
	if (on){
		gpio_direction_output(MXS_IOMUX_PAD(3,4,PAD_MUXSEL_GPIO,0,0,0),0);
		gpio_direction_output(MXS_IOMUX_PAD(3,5,PAD_MUXSEL_GPIO,0,0,0),0);
	}
	else{
		gpio_direction_input(MXS_IOMUX_PAD(3,4,PAD_MUXSEL_GPIO,0,0,0));
		gpio_direction_input(MXS_IOMUX_PAD(3,5,PAD_MUXSEL_GPIO,0,0,0));
	}
}
/*
 * check whether battery temperature in valid range for charging
 * Returns: 1: valid; 0: invalid
 */
/* in NTC-20 EVT3 it is only possible to measure high threshold
 * in DVT we can measure low threshold
 */
#if defined(HW_EVT3)
/* threshold on battery temperature monitor voltage (in mV), with current 20uA, indicating over-heat for charging */
/* Below values refer to datasheet of Chip Type NTC Thermistor CN0603R104B4100JT */
/* 60 degree C, R=23.4988 KOhm */
#define BATT_TEMP_VOLT_THRESH 470
#else
/* threshold on battery temperature monitor voltage (in mV), with current 20uA, indicating over-heat for charging */
/* Below values refer to datasheet of Chip Type NTC Thermistor CN0603R104B4100JT */
/* 60 degree C, Rt=23.4988 KOhm, R_total = (Rt*100)/(Rt+100) = 19.027553304 KOhm; V=I*R */
#define BATT_TEMP_VOLT_THRESH 381

/* voltage threshold (in MV) corresponding to minimum temperature allowed for charging
 * 0 degree C, Rt = 346.6868 KOhm
 */
#define MIN_BATT_TEMP_CHARGING_MON_VOLT 1552
#endif

int mx28_is_batt_temp_ok_for_charging(void)
{
	int batt_temp_volt = 0;
	int result = 0;
	/* turn on current */
	mx28_turn_temp_current_source(LRADC0,1);
	udelay(DELAY_TO_CHECK_BATTERY_US);

#if defined(HW_EVT3)
	/* measure volt at BATT_TEMP_MON */
	batt_temp_volt = mx28_measure_lradc(LRADC0,0);
	if (batt_temp_volt>=BATT_TEMP_VOLT_THRESH){
		result = 1;
	}
	else{
		result = 0;
	}
#else
	batt_temp_volt = mx28_measure_lradc(LRADC0,1);
	if (batt_temp_volt>=BATT_TEMP_VOLT_THRESH && batt_temp_volt<=MIN_BATT_TEMP_CHARGING_MON_VOLT){
		result = 1;
	}
	else{
		result = 0;
	}
#endif
	/* turn off current */
	mx28_turn_temp_current_source(LRADC0,0);
	return result;
}
/* power down the chip */
void mx28_powerdown_the_chip(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	/* make sure CELL_EXT_POWER_EN is not output low, otherwise RESET is low and
	 * it won't start when VDD5V is available
	 */
	gpio_direction_input(MXS_IOMUX_PAD(4,4,PAD_MUXSEL_GPIO,0,0,0));
	/* power down */
	writel(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD, &power_regs->hw_power_reset);
}
/*
 * check power state
 * Parameters:
 * 	state:	pointer to struct power_state (must already allocated) to fill in values
 */
void mx28_check_power_state(struct power_status *state)
{
	int ac_sense = 0, charger_being_on = 0;
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	if (!state){
		return;
	}
	/* check 5V */
	state->vdd5v_detected = (readl(&power_regs->hw_power_sts) & POWER_STS_VBUSVALID0)?1:0;
	/* check charger or usb host */
	if (state->vdd5v_detected){
		ac_sense = mx28_measure_lradc(LRADC_AC_IN_MON,1);
		D("ac_sense=%d \r\n",ac_sense);
		if ((ac_sense>=LRADC_5V_THRESHOLD) || !mx28_is_usb_data_signal_detected()){
			/* AC or USB charger*/
			state->vdd5v_charger=1;
		}
		else{
			/* USB host */
			state->vdd5v_charger=0;
		}
	}
	else{
		state->vdd5v_charger=0;
	}
	/* check battery */
	/* battery charger must be turn off here to check battery */
	if (readl(&power_regs->hw_power_charge) & POWER_CHARGE_PWD_BATTCHRG){
		charger_being_on = 0;
	}
	else{
		writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_set);
		charger_being_on = 1;
		udelay(DELAY_TO_CHECK_BATTERY_US);
	}
	state->batt_detected = mx28_is_batt_detected();
	if (state->batt_detected){
		state->batt_ok_to_boot = mx28_is_batt_ok_to_boot();
		D("Battery detected, battery voltage=%d\r\n",mx28_get_batt_volt());
	}
	else{
		state->batt_ok_to_boot = 0;
	}
	/* if battery charger is on previously, turn on here, then add some delay */
	if (charger_being_on){
		writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_clr);
		udelay(DELAY_TO_CHECK_BATTERY_US);
	}
}


/* check pswitch level change
 * Return: 1: pswitch level changed; 0: no change
 */
int mx28_pswitch_event(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	int pswitch_mid = 0;
	pswitch_mid = (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_PSWITCH_IRQ)?1:0;
	if (pswitch_mid){
		/* reset bit */
		writel(POWER_CTRL_PSWITCH_IRQ, &power_regs->hw_power_ctrl_clr);
	}
	return pswitch_mid;
}
/* check vdd5v
 * Returns: 1: detected; 0: not detected
 */
int mx28_is_vdd5v_detected(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	return ((readl(&power_regs->hw_power_sts) & POWER_STS_VBUSVALID0)?1:0);
}
/* check whether charger is on
 * Return: 1: ON; 0: OFF
 */
int mx28_is_charger_on(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	return ((readl(&power_regs->hw_power_charge) & POWER_CHARGE_PWD_BATTCHRG) == 0);
}
/* turn off charger */
void mx28_turn_off_charger(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	if (mx28_is_charger_on()){
		/* stop charging */
		writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_set);
		writel(POWER_CHARGE_STOP_ILIMIT_MASK | POWER_CHARGE_BATTCHRG_I_MASK, &power_regs->hw_power_charge_clr);
	}
}

/*
 * turn on charger
 *
 * Parameter stop_ilimit and charge_ilimit represent HW_POWER_CHARGE[STOP_ILIMIT] and
 *  HW_POWER_CHARGE[BATTCHRG_I] respectively.
 *  - HW_POWER_CHARGE[STOP_ILIMIT]: Current threshold at which the Li-Ion battery charger signals to stop charging.
 *  The current represented by each bit is as follows: (100 mA, 50 mA, 20 mA, 10 mA) = (bit 3, bit 2, bit 1, bit 0).
 *  - HW_POWER_CHARGE[BATTCHRG_I]: Magnitude of the battery charge current, the current represented by each bit is
 *   as follows: (400 mA, 200mA, 100 mA, 50 mA, 20 mA, 10 mA) = (bit 5,bit 4, bit 3, bit 2, bit 1, bit 0)
 */
void mx28_turn_on_charger(int stop_ilimit, int charge_ilimit)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	clrsetbits_le32(&power_regs->hw_power_charge,
		POWER_CHARGE_STOP_ILIMIT_MASK | POWER_CHARGE_BATTCHRG_I_MASK,
		stop_ilimit | charge_ilimit);
	writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_clr);
}
/* check whether charge current is below stop ilimit and if it is, turn off charger */
void mx28_check_charge_ilimit_stop(void)
{
#define LOOP_CHECKING_ILIMIT_STOP 5
#define DELAY_TO_CHECK_ILIMIT_STOP 20
	int current_below_stop_limit = 0;
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	/* if charger is on, check ilimit stop */
	if (mx28_is_charger_on()){
		while ((readl( &power_regs->hw_power_sts ) & POWER_STS_CHRGSTS) == 0 &&
				current_below_stop_limit < LOOP_CHECKING_ILIMIT_STOP ){
			current_below_stop_limit++;
			udelay(DELAY_TO_CHECK_ILIMIT_STOP);
		}
		/* if ilimit stop is confirmed, turn off charger */
		if ((readl( &power_regs->hw_power_sts ) & POWER_STS_CHRGSTS) == 0 &&
				current_below_stop_limit == LOOP_CHECKING_ILIMIT_STOP){
			writel(POWER_CHARGE_PWD_BATTCHRG, &power_regs->hw_power_charge_set);
		}
		current_below_stop_limit = 0;
	}
}
/* check whether battery voltage is below high threshold
 * Return: 1: Yes; 0: No
 */
int mx28_is_batt_volt_below_high_thresh(void)
{
	return (mx28_get_batt_volt() <= BATT_VOLT_THRESH_REENABLE_CHARGER);
}
/*
 * disable pswitch irq
 */
void mx28_disable_pswitch_irq(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	writel(POWER_CTRL_PSWITCH_MID_TRAN | POWER_CTRL_ENIRQ_PSWITCH, &power_regs->hw_power_ctrl_clr);
}
/* hand over to battery */
void mx28_hand_over_battery(void)
{
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	int val;

	/* turn off DCDC4P2 */
	val = readl(&power_regs->hw_power_dcdc4p2);
	val &= ~(POWER_DCDC4P2_ENABLE_4P2 | POWER_DCDC4P2_ENABLE_DCDC);
	writel(val, &power_regs->hw_power_dcdc4p2);
	/* power down 4p2 and charger */
	writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK, &power_regs->hw_power_5vctrl_set);
	/* Allow DCDC be to active when 5V is present. */
	writel(POWER_5VCTRL_ENABLE_DCDC, &power_regs->hw_power_5vctrl_set);
	/*
	 * Sets the trip point for the comparison between the DCDC_4P2 and BATTERY pin
	 * 0b11111 DCDC_4P2 pin >= 1.05 * BATTERY pin
	 */
	writel(readl(&power_regs->hw_power_dcdc4p2) | POWER_DCDC4P2_CMPTRIP_MASK, &power_regs->hw_power_dcdc4p2);
}
/* state machine */
power_state_machine_t state_machine = OFF;
/*
 * Phase 1 power state machine
 * - setup pswitch irq
 * - Get user request:
 * 		+ shutdown or low battery event:
 * 			* battery only: shutdown
 * 			* AC/USB charger/USB host:
 * 					SHUTDOWN: loop waiting for pswitch event, charge battery if necessary
 * 					LOW BATTERY: continue booting
 * 		+ no user request: return to continue booting
 */
static void mx28_spl_phase_1_power_state_machine(void)
{
	int pswitch_mid = 0, user_request = 0;
	struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
	struct mx28_rtc_regs *rtc_regs = (struct mx28_rtc_regs *)MXS_RTC_BASE;
	int led_is_on = 1;
	int charge_ilimit = 0;
	uint32_t watchdog_timer_backup;

	struct power_status pw_state;
	/* global variable state_machine may keep old value from previous run */
	state_machine = OFF;
	/* setup pswitch irq */
	writel(POWER_CTRL_PSWITCH_MID_TRAN | POWER_CTRL_ENIRQ_PSWITCH, &power_regs->hw_power_ctrl_set);
	mx28_enable_usb_plugin_detect();
	/* get user request */
    user_request = (readl(&rtc_regs->hw_rtc_persistent5) & BM_REQUEST_PERSISTENT);
    /* processing for user request SHUTDOWN and LOW BATTERY:
     * 1) If battery only: shutdown
     * 2) If vdd5v:
     * 		SHUTDOWN: loop waiting for pswitch event
     * 		LOW BATTERY: go to phase 2
     */
    if ((user_request == LOW_BATT_REQUEST_PERSISTENT) || (user_request == SHUTDOWN_REQUEST_PERSISTENT)){
    	/* clear user request */
    	writel(BM_REQUEST_PERSISTENT, &rtc_regs->hw_rtc_persistent5_clr);
		/* check 5V */
    	pw_state.vdd5v_detected = mx28_is_vdd5v_detected();
		if (pw_state.vdd5v_detected){
			if (user_request == SHUTDOWN_REQUEST_PERSISTENT){
				/* loop until pswitch event, charging if necessary */
				/* backup watchdog time */
				watchdog_timer_backup = readl(&rtc_regs->hw_rtc_watchdog);
				/* loop until pswitch event */
				while (state_machine != PSWITCH_MID)
				{
					/* kick watch dog */
					writel(WATCHDOG_TIMER_STATE_MACHINE_MS, &rtc_regs->hw_rtc_watchdog);/*ms */
					/* update power state */
					mx28_check_power_state(&pw_state);
					/* check pswitch level change */
					pswitch_mid = mx28_pswitch_event();
					D("Phase 1 State machine:: state_machine=%d, vdd5v_detected=%d, vdd5v_charger=%d, pswitch_mid=%d, batt_detected=%d, batt_ok_to_boot=%d \r\n",
							state_machine, pw_state.vdd5v_detected, pw_state.vdd5v_charger, pswitch_mid, pw_state.batt_detected, pw_state.batt_ok_to_boot);

					if (pswitch_mid){
						/* pswitch event, turn off charger if any and let's go */
						mx28_turn_off_charger();
						state_machine = PSWITCH_MID;
					}
					else{
						if(pw_state.vdd5v_detected){
							/* charging if necessary */
							if (pw_state.batt_detected && mx28_is_batt_temp_ok_for_charging()){
								/* if charger is on, check ilimit stop */
								if (mx28_is_charger_on()){
									mx28_check_charge_ilimit_stop();
								}
								/* else: check battery voltage threshold to re-enable charger */
								else {
									if (mx28_is_batt_volt_below_high_thresh()){
										/* start charging */
										if (pw_state.vdd5v_charger){
											/* 320 mA */
											charge_ilimit = POWER_CHARGE_BATTCHRG_I_200MA | POWER_CHARGE_BATTCHRG_I_100MA | POWER_CHARGE_BATTCHRG_I_20MA;
										}
										else{
											/* 80mA */
											charge_ilimit = POWER_CHARGE_BATTCHRG_I_50MA | POWER_CHARGE_BATTCHRG_I_20MA | POWER_CHARGE_BATTCHRG_I_10MA;
										}
										mx28_turn_on_charger(POWER_CHARGE_STOP_ILIMIT_10MA, charge_ilimit);
									}
								}
							} /* batt detected and temp ok */
							else{
								/* turn off charger */
								mx28_turn_off_charger();
							}
						}
						else{
							/* no vdd5v, shutdown */
							mx28_powerdown_the_chip();
						}
					}
					if (state_machine != PSWITCH_MID){
						/* turn off led if staying in OFF state */
						if (led_is_on){
							mx28_turn_on_led(0);
							led_is_on = 0;
						}
						udelay(DELAY_FOR_STATE_MACHINE_CYCLE_US);
					}
				}
				D("Phase 1 state machine:: state_machine=%d, continue... \r\n", state_machine);
				/* restore watchdog time */
				writel(watchdog_timer_backup, &rtc_regs->hw_rtc_watchdog);
			}
			else{
				/* in the case of low battery request, go to phase 2 */
				state_machine = PSWITCH_MID;
			}
		}
		/* if no vdd5v detected, shutdown */
		else{
			mx28_powerdown_the_chip();
		}
    }
    else{
    	/* if no user request, just go to phase 2 */
    	state_machine = PSWITCH_MID;
    }
    /* turn on led */
    if (!led_is_on){
    	mx28_turn_on_led(1);
    }
}
#endif

#if defined(CONFIG_UNIFIED_POWER_INIT_PATH)
static void mx28_boot_unified(void)
{
    struct mx28_power_regs *power_regs = (struct mx28_power_regs *)MXS_POWER_BASE;
    uint32_t tmp, tmp2, vbus_thresh, vbus_5vdetect, pwd_bo;

    clrsetbits_le32(&power_regs->hw_power_minpwr,
    			POWER_MINPWR_HALFFETS, POWER_MINPWR_DOUBLE_FETS);

    mx28_power_set_linreg();


    /* Setup 4P2 parameters */

    /* Set Regulation voltage of the DCDC_4P2 pin: 4.2V.*/
    /* Sets the trip point for the comparison between the DCDC_4P2 and BATTERY pin: 0b11111 DCDC_4P2 pin >= 1.05 * BATTERY pin*/
    clrsetbits_le32(&power_regs->hw_power_dcdc4p2,
    		POWER_DCDC4P2_CMPTRIP_MASK | POWER_DCDC4P2_TRG_MASK,
    		POWER_DCDC4P2_TRG_4V2 | (POWER_DCDC4P2_CMPTRIP_MASK << POWER_DCDC4P2_CMPTRIP_OFFSET));

    /* set VDD4P2 brownout 100mV below the target set by DCDC4p2_trg before the regulation circuit steals battery charge current to support the voltage on VDD4P2 */
    /* set DcDc converter selects either VDD4P2 or BATTERY, which ever is higher. */
    clrsetbits_le32(&power_regs->hw_power_dcdc4p2,
    		POWER_DCDC4P2_DROPOUT_CTRL_MASK,
    		POWER_DCDC4P2_DROPOUT_CTRL_100MV |
    		POWER_DCDC4P2_DROPOUT_CTRL_SRC_SEL);


    /* enable 4p2 */
	setbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_ENABLE_4P2);

	writel(POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK,
		&power_regs->hw_power_5vctrl_clr);
	clrbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_TRG_MASK);

	/* Power up the 4p2 rail and logic/control */
	writel(POWER_5VCTRL_PWD_CHARGE_4P2_MASK,
		&power_regs->hw_power_5vctrl_clr);

	clrbits_le32(&power_regs->hw_power_5vctrl, POWER_5VCTRL_PWDN_5VBRNOUT);

	/*disable all paths to power off the chip except the watchdog timer */
	writel(POWER_RESET_UNLOCK_KEY | POWER_RESET_PWD_OFF, &power_regs->hw_power_reset);

	clrbits_le32(&power_regs->hw_power_ctrl, POWER_CTRL_ENIRQ_VDD5V_DROOP);

	/*
	 * Recording orignal values that will be modified temporarlily
	 * to handle a chip bug. See chip errata for CQ ENGR00115837
	 */
	tmp = readl(&power_regs->hw_power_5vctrl);
	vbus_thresh = tmp & POWER_5VCTRL_VBUSVALID_TRSH_MASK;
	vbus_5vdetect = tmp & POWER_5VCTRL_VBUSVALID_5VDETECT;

	/* backup pwd bo setting */
	pwd_bo = readl(&power_regs->hw_power_minpwr) & POWER_MINPWR_PWD_BO;

	/*
	 * Disable mechanisms that get erroneously tripped by when setting
	 * the DCDC4P2 EN_DCDC
	 */
	clrbits_le32(&power_regs->hw_power_5vctrl,
		POWER_5VCTRL_VBUSVALID_5VDETECT |
		POWER_5VCTRL_VBUSVALID_TRSH_MASK);

	/* Powers down supply brownout comparators. */
	writel(POWER_MINPWR_PWD_BO, &power_regs->hw_power_minpwr_set);


	setbits_le32(&power_regs->hw_power_dcdc4p2, POWER_DCDC4P2_ENABLE_DCDC);

	early_delay(DELAY_BEFORE_RESTORE_SETTINGS_US);

	/* restore settings */
	clrsetbits_le32(&power_regs->hw_power_5vctrl,
			POWER_5VCTRL_VBUSVALID_TRSH_MASK, vbus_thresh);

	if (vbus_5vdetect)
		writel(vbus_5vdetect, &power_regs->hw_power_5vctrl_set);

	if (!pwd_bo)
		clrbits_le32(&power_regs->hw_power_minpwr, POWER_MINPWR_PWD_BO);

	/* clear irq status bit.  */
	while (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VBUS_VALID_IRQ)
		writel(POWER_CTRL_VBUS_VALID_IRQ,
			&power_regs->hw_power_ctrl_clr);

	while (readl(&power_regs->hw_power_ctrl) & POWER_CTRL_VDD5V_DROOP_IRQ)
		writel(POWER_CTRL_VDD5V_DROOP_IRQ,
			&power_regs->hw_power_ctrl_clr);


	/*
	 * ramp up the current limit
	 * so that we don't cause large inrush current issues. We step up the
	 * current limit until until we've
	 * reached our maximum defined 4p2 current limit.
	 */

	tmp = 0;
	while (tmp < AC_CHARGE_4P2_ILIMIT) {
		tmp++;
		tmp2 = readl(&power_regs->hw_power_5vctrl);
		tmp2 &= ~POWER_5VCTRL_CHARGE_4P2_ILIMIT_MASK;
		tmp2 |= tmp << POWER_5VCTRL_CHARGE_4P2_ILIMIT_OFFSET;
		writel(tmp2, &power_regs->hw_power_5vctrl);
		early_delay(RAMP_UP_CURRENT_LIMIT_TIME_US);
	}

    /* clear dcdc4p2 brownout status, disable 4p2 brownout interrupt */
    clrbits_le32(&power_regs->hw_power_dcdc4p2,
    				POWER_DCDC4P2_BO_MASK);
    writel(POWER_CTRL_DCDC4P2_BO_IRQ,
    				&power_regs->hw_power_ctrl_clr);
    writel(POWER_CTRL_ENIRQ_DCDC4P2_BO,
    				&power_regs->hw_power_ctrl_clr);

    clrbits_le32(&power_regs->hw_power_vdddctrl,
    		POWER_VDDDCTRL_DISABLE_FET | POWER_VDDDCTRL_ENABLE_LINREG);

    clrbits_le32(&power_regs->hw_power_vddactrl,
    		POWER_VDDACTRL_DISABLE_FET | POWER_VDDACTRL_ENABLE_LINREG);

    clrbits_le32(&power_regs->hw_power_vddioctrl,
    		POWER_VDDIOCTRL_DISABLE_FET);

    setbits_le32(&power_regs->hw_power_5vctrl,
    		POWER_5VCTRL_ENABLE_DCDC);
}
#endif

static void mx28_power_configure_power_source(void)
{
#ifndef CONFIG_UNIFIED_POWER_INIT_PATH
#ifndef CONFIG_NO_VDD5V
	int batt_ready, batt_good;
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	struct mx28_lradc_regs *lradc_regs =
		(struct mx28_lradc_regs *)MXS_LRADC_BASE;
#endif

	mx28_src_power_init();

#if defined(CONFIG_NO_VDD5V)
	mx28_batt_boot();
#else
	batt_ready = mx28_is_batt_ready();

	if (readl(&power_regs->hw_power_sts) & POWER_STS_VDD5V_GT_VDDIO) {
		batt_good = mx28_is_batt_good();
		if (batt_ready) {
			/* 5V source detected, good battery detected. */
			mx28_batt_boot();
		} else {
			if (batt_good) {
				/* 5V source detected, low battery detceted. */
			} else {
				/* 5V source detected, bad battery detected. */
				writel(LRADC_CONVERSION_AUTOMATIC,
					&lradc_regs->hw_lradc_conversion_clr);
				clrbits_le32(&power_regs->hw_power_battmonitor,
					POWER_BATTMONITOR_BATT_VAL_MASK);
			}
			mx28_5v_boot();
		}
	} else {
		/* 5V not detected, booting from battery. */
		mx28_batt_boot();
	}
#endif /* CONFIG_NO_VDD5V */

	mx28_init_batt_bo();

	mx28_switch_vddd_to_dcdc_source();
#else
	mx28_src_power_init();

	mx28_boot_unified();

	mx28_switch_vddd_to_dcdc_source();
#endif
}

static void mx28_enable_output_rail_protection(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	writel(POWER_CTRL_VDDD_BO_IRQ | POWER_CTRL_VDDA_BO_IRQ |
		POWER_CTRL_VDDIO_BO_IRQ, &power_regs->hw_power_ctrl_clr);

	/*
	 * MJC - Disabled this as this was preventing the MICE board from
	 * booting, since this is the 'production' spl and is only used
	 * during device production (not normal use) I'm not going to give this
	 * any more thought.
	 */

	D("Not Enabling output rail protection !!\r\n");
//	setbits_le32(&power_regs->hw_power_vdddctrl,
//			POWER_VDDDCTRL_PWDN_BRNOUT);

//	setbits_le32(&power_regs->hw_power_vddactrl,
//			POWER_VDDACTRL_PWDN_BRNOUT);

//	setbits_le32(&power_regs->hw_power_vddioctrl,
//			POWER_VDDIOCTRL_PWDN_BRNOUT);
}

static int mx28_get_vddio_power_source_off(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t tmp;

	if (readl(&power_regs->hw_power_sts) & POWER_STS_VDD5V_GT_VDDIO) {
		tmp = readl(&power_regs->hw_power_vddioctrl);
		if (tmp & POWER_VDDIOCTRL_DISABLE_FET) {
			if ((tmp & POWER_VDDIOCTRL_LINREG_OFFSET_MASK) ==
				POWER_VDDDCTRL_LINREG_OFFSET_0STEPS) {
				return 1;
			}
		}

		if (!(readl(&power_regs->hw_power_5vctrl) &
			POWER_5VCTRL_ENABLE_DCDC)) {
			if ((tmp & POWER_VDDIOCTRL_LINREG_OFFSET_MASK) ==
				POWER_VDDDCTRL_LINREG_OFFSET_0STEPS) {
				return 1;
			}
		}
	}

	return 0;

}

static int mx28_get_vddd_power_source_off(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t tmp;

	tmp = readl(&power_regs->hw_power_vdddctrl);
	if (tmp & POWER_VDDDCTRL_DISABLE_FET) {
		if ((tmp & POWER_VDDDCTRL_LINREG_OFFSET_MASK) ==
			POWER_VDDDCTRL_LINREG_OFFSET_0STEPS) {
			return 1;
		}
	}

	if (readl(&power_regs->hw_power_sts) & POWER_STS_VDD5V_GT_VDDIO) {
		if (!(readl(&power_regs->hw_power_5vctrl) &
			POWER_5VCTRL_ENABLE_DCDC)) {
			return 1;
		}
	}

	if (!(tmp & POWER_VDDDCTRL_ENABLE_LINREG)) {
		if ((tmp & POWER_VDDDCTRL_LINREG_OFFSET_MASK) ==
			POWER_VDDDCTRL_LINREG_OFFSET_1STEPS_BELOW) {
			return 1;
		}
	}

	return 0;
}

void mx28_power_set_vddio(uint32_t new_target, uint32_t new_brownout)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t cur_target, diff, bo_int = 0;
	uint32_t powered_by_linreg = 0;

	new_brownout = new_target - new_brownout;

	cur_target = readl(&power_regs->hw_power_vddioctrl);
	cur_target &= POWER_VDDIOCTRL_TRG_MASK;
	cur_target *= 50;	/* 50 mV step*/
	cur_target += 2800;	/* 2800 mV lowest */

	powered_by_linreg = mx28_get_vddio_power_source_off();
	if (new_target > cur_target) {

		if (powered_by_linreg) {
			bo_int = readl(&power_regs->hw_power_vddioctrl);
			clrbits_le32(&power_regs->hw_power_vddioctrl,
					POWER_CTRL_ENIRQ_VDDIO_BO);
		}

		setbits_le32(&power_regs->hw_power_vddioctrl,
				POWER_VDDIOCTRL_BO_OFFSET_MASK);
		do {
			if (new_target - cur_target > 100)
				diff = cur_target + 100;
			else
				diff = new_target;

			diff -= 2800;
			diff /= 50;

			clrsetbits_le32(&power_regs->hw_power_vddioctrl,
				POWER_VDDIOCTRL_TRG_MASK, diff);

			if (powered_by_linreg ||
				(readl(&power_regs->hw_power_sts) &
					POWER_STS_VDD5V_GT_VDDIO))
				early_delay(500);
			else {
				while (!(readl(&power_regs->hw_power_sts) &
					POWER_STS_DC_OK))
					;

			}

			cur_target = readl(&power_regs->hw_power_vddioctrl);
			cur_target &= POWER_VDDIOCTRL_TRG_MASK;
			cur_target *= 50;	/* 50 mV step*/
			cur_target += 2800;	/* 2800 mV lowest */
		} while (new_target > cur_target);

		if (powered_by_linreg) {
			writel(POWER_CTRL_VDDIO_BO_IRQ,
				&power_regs->hw_power_ctrl_clr);
			if (bo_int & POWER_CTRL_ENIRQ_VDDIO_BO)
				setbits_le32(&power_regs->hw_power_vddioctrl,
						POWER_CTRL_ENIRQ_VDDIO_BO);
		}
	} else {
		do {
			if (cur_target - new_target > 100)
				diff = cur_target - 100;
			else
				diff = new_target;

			diff -= 2800;
			diff /= 50;

			clrsetbits_le32(&power_regs->hw_power_vddioctrl,
				POWER_VDDIOCTRL_TRG_MASK, diff);

			if (powered_by_linreg ||
				(readl(&power_regs->hw_power_sts) &
					POWER_STS_VDD5V_GT_VDDIO))
				early_delay(500);
			else {
				while (!(readl(&power_regs->hw_power_sts) &
					POWER_STS_DC_OK))
					;

			}

			cur_target = readl(&power_regs->hw_power_vddioctrl);
			cur_target &= POWER_VDDIOCTRL_TRG_MASK;
			cur_target *= 50;	/* 50 mV step*/
			cur_target += 2800;	/* 2800 mV lowest */
		} while (new_target < cur_target);
	}

	clrsetbits_le32(&power_regs->hw_power_vddioctrl,
			POWER_VDDDCTRL_BO_OFFSET_MASK,
			new_brownout << POWER_VDDDCTRL_BO_OFFSET_OFFSET);
}

static void mx28_power_set_vddd(uint32_t new_target, uint32_t new_brownout)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t cur_target, diff, bo_int = 0;
	uint32_t powered_by_linreg = 0;

	new_brownout = new_target - new_brownout;

	cur_target = readl(&power_regs->hw_power_vdddctrl);
	cur_target &= POWER_VDDDCTRL_TRG_MASK;
	cur_target *= 25;	/* 25 mV step*/
	cur_target += 800;	/* 800 mV lowest */

	powered_by_linreg = mx28_get_vddd_power_source_off();
	if (new_target > cur_target) {
		if (powered_by_linreg) {
			bo_int = readl(&power_regs->hw_power_vdddctrl);
			clrbits_le32(&power_regs->hw_power_vdddctrl,
					POWER_CTRL_ENIRQ_VDDD_BO);
		}

		setbits_le32(&power_regs->hw_power_vdddctrl,
				POWER_VDDDCTRL_BO_OFFSET_MASK);

		do {
			if (new_target - cur_target > 100)
				diff = cur_target + 100;
			else
				diff = new_target;

			diff -= 800;
			diff /= 25;

			clrsetbits_le32(&power_regs->hw_power_vdddctrl,
				POWER_VDDDCTRL_TRG_MASK, diff);

			if (powered_by_linreg ||
				(readl(&power_regs->hw_power_sts) &
					POWER_STS_VDD5V_GT_VDDIO))
				early_delay(500);
			else {
				while (!(readl(&power_regs->hw_power_sts) &
					POWER_STS_DC_OK))
					;

			}

			cur_target = readl(&power_regs->hw_power_vdddctrl);
			cur_target &= POWER_VDDDCTRL_TRG_MASK;
			cur_target *= 25;	/* 25 mV step*/
			cur_target += 800;	/* 800 mV lowest */
		} while (new_target > cur_target);

		if (powered_by_linreg) {
			writel(POWER_CTRL_VDDD_BO_IRQ,
				&power_regs->hw_power_ctrl_clr);
			if (bo_int & POWER_CTRL_ENIRQ_VDDD_BO)
				setbits_le32(&power_regs->hw_power_vdddctrl,
						POWER_CTRL_ENIRQ_VDDD_BO);
		}
	} else {
		do {
			if (cur_target - new_target > 100)
				diff = cur_target - 100;
			else
				diff = new_target;

			diff -= 800;
			diff /= 25;

			clrsetbits_le32(&power_regs->hw_power_vdddctrl,
					POWER_VDDDCTRL_TRG_MASK, diff);

			if (powered_by_linreg ||
				(readl(&power_regs->hw_power_sts) &
					POWER_STS_VDD5V_GT_VDDIO))
				early_delay(500);
			else {
				while (!(readl(&power_regs->hw_power_sts) &
					POWER_STS_DC_OK))
					;

			}

			cur_target = readl(&power_regs->hw_power_vdddctrl);
			cur_target &= POWER_VDDDCTRL_TRG_MASK;
			cur_target *= 25;	/* 25 mV step*/
			cur_target += 800;	/* 800 mV lowest */
		} while (new_target < cur_target);
	}

	clrsetbits_le32(&power_regs->hw_power_vdddctrl,
			POWER_VDDDCTRL_BO_OFFSET_MASK,
			new_brownout << POWER_VDDDCTRL_BO_OFFSET_OFFSET);
}

static void mx28_setup_batt_detect(void)
{
	mx28_lradc_init();
	mx28_lradc_enable_batt_measurement();
	early_delay(10);
}
#ifdef PRINT_SOME_REGISTERS
#define PR_R(n) \
	app_printf(#n " 0x%8x\n\r", readl(&base->n))

static void pr_power(void)
{
	struct mx28_power_regs *base =
		(struct mx28_power_regs *)MXS_POWER_BASE;
        PR_R(hw_power_ctrl);
        PR_R(hw_power_5vctrl);
        PR_R(hw_power_minpwr);
        PR_R(hw_power_charge);
        PR_R(hw_power_vdddctrl);
        PR_R(hw_power_vddactrl);
        PR_R(hw_power_vddioctrl);
        PR_R(hw_power_vddmemctrl);
        PR_R(hw_power_dcdc4p2);
        PR_R(hw_power_misc);
        PR_R(hw_power_dclimits);
        PR_R(hw_power_loopctrl);
        PR_R(hw_power_sts);
        PR_R(hw_power_speed);
        PR_R(hw_power_battmonitor);
        PR_R(hw_power_reset);
        PR_R(hw_power_debug);
        PR_R(hw_power_thermal);
        PR_R(hw_power_usb1ctrl);
        PR_R(hw_power_special);
        PR_R(hw_power_version);
        PR_R(hw_power_anaclkctrl);
        PR_R(hw_power_refctrl);
}

static void pr_clock(void)
{
	struct mx28_clkctrl_regs *base =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
        PR_R(hw_clkctrl_pll0ctrl0);
        PR_R(hw_clkctrl_pll0ctrl1);
        PR_R(hw_clkctrl_pll1ctrl0);
        PR_R(hw_clkctrl_pll1ctrl1);
        PR_R(hw_clkctrl_pll2ctrl0);
        PR_R(hw_clkctrl_cpu);
        PR_R(hw_clkctrl_hbus);
        PR_R(hw_clkctrl_xbus);
        PR_R(hw_clkctrl_xtal);
        PR_R(hw_clkctrl_ssp0);
        PR_R(hw_clkctrl_ssp1);
        PR_R(hw_clkctrl_ssp2);
        PR_R(hw_clkctrl_ssp3);
        PR_R(hw_clkctrl_gpmi);
        PR_R(hw_clkctrl_spdif);
        PR_R(hw_clkctrl_emi);
        PR_R(hw_clkctrl_saif0);
        PR_R(hw_clkctrl_saif1);
        PR_R(hw_clkctrl_lcdif);
        PR_R(hw_clkctrl_etm);
        PR_R(hw_clkctrl_enet);
        PR_R(hw_clkctrl_hsadc);
        PR_R(hw_clkctrl_flexcan);
        PR_R(hw_clkctrl_frac0);
        PR_R(hw_clkctrl_frac1);
        PR_R(hw_clkctrl_clkseq);
        PR_R(hw_clkctrl_reset);
        PR_R(hw_clkctrl_status);
        PR_R(hw_clkctrl_version);
}

static void pr_rtc(void)
{
	struct mx28_rtc_regs *base =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
        PR_R(hw_rtc_ctrl);
        PR_R(hw_rtc_stat);
        PR_R(hw_rtc_milliseconds);
        PR_R(hw_rtc_seconds);
        PR_R(hw_rtc_rtc_alarm);
        PR_R(hw_rtc_watchdog);
        PR_R(hw_rtc_persistent0);
        PR_R(hw_rtc_persistent1);
        PR_R(hw_rtc_persistent2);
        PR_R(hw_rtc_persistent3);
        PR_R(hw_rtc_persistent4);
        PR_R(hw_rtc_persistent5);
        PR_R(hw_rtc_debug);
        PR_R(hw_rtc_version);
}

static void pr_persist(void)
{
	struct mx28_rtc_regs *base =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
        PR_R(hw_rtc_persistent0);
}
#endif
static void mx28_setup_minpower(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
#if 0
	setbits_le32(&power_regs->hw_power_minpwr,
	/* */		// POWER_MINPWR_DOUBLE_FETS|
	/*set */	POWER_MINPWR_LOWPWR_4P2|
			POWER_MINPWR_PWD_BO|
			POWER_MINPWR_USE_VDDXTAL_VBG|
			POWER_MINPWR_PWD_ANA_CMPS|
			POWER_MINPWR_ENABLE_OSC|
			POWER_MINPWR_SELECT_OSC|
			POWER_MINPWR_LESSANA_I|
			// POWER_MINPWR_HALFFETS|
			POWER_MINPWR_EN_DC_PFM|
			// POWER_MINPWR_DC_HALFCLK|
			0
	);
	setbits_le32(&power_regs->hw_power_minpwr,
			POWER_MINPWR_PWD_XTAL24);
#endif
	setbits_le32(&power_regs->hw_power_minpwr,
	/* */		// POWER_MINPWR_DOUBLE_FETS|
	/*set */
			POWER_MINPWR_EN_DC_PFM|
			POWER_MINPWR_DC_HALFCLK|
			0
	);
	setbits_le32(&power_regs->hw_power_loopctrl, POWER_LOOPCTRL_HYST_SIGN);
}

#define MX28_HW_DIGCTL_MICROSECONDS     0x8001c0c0

/* mx28_sft_reset_block and mx28_release_reset_block are not used by any code.
 * keep here for reference
 */
#if 0
static void mx28_sft_reset_block(unsigned *reg)
{
unsigned long start;
	// Prepare for soft-reset by making sure that SFTRST is not currently
	// asserted. Also clear CLKGATE so we can wait for its assertion below.
	clrbits_le32(reg, (1<<31));
	// Wait at least a microsecond for SFTRST to deassert. In actuality, we
	// need to wait 3 GPMI clocks, but this is much easier to implement.
	start = readl(MX28_HW_DIGCTL_MICROSECONDS);
	while ((readl(reg) & (1<<31)) || readl(MX28_HW_DIGCTL_MICROSECONDS) - start < 2)
		{}
	// Also clear CLKGATE so we can wait for its assertion below.
	clrbits_le32(reg, (1<<30));
	// Now soft-reset the hardware.
	setbits_le32(reg, (1<<31));
	// Poll until clock is in the gated state before subsequently
	// clearing soft reset and clock gate.
	while (!(readl(reg) & (1<<30)))
	{
	; // busy wait
	}
}

static void mx28_release_reset_block(unsigned *reg)
{
unsigned long start;
	// Prepare for soft-reset by making sure that SFTRST is not currently
	// bring GPMI_CTRL0 out of reset
	clrbits_le32(reg, (1<<31));
	// Wait at least a microsecond for SFTRST to deassert. In actuality, we
	// need to wait 3 GPMI clocks, but this is much easier to implement.
	start = readl(MX28_HW_DIGCTL_MICROSECONDS);
	while ((readl(reg) & (1<<31)) || readl(MX28_HW_DIGCTL_MICROSECONDS) - start < 2)
		{}
	clrbits_le32(reg, (1<<30));
	// Poll until clock is in the NON-gated state before returning.
	while (readl(reg) & (1<<30))
	{
	; // busy wait
	}
}
#endif

#if 0
MXS_ICOL_BASE           //0x80000020 ?
MXS_HSADC_BASE          //0x80002000 *
MXS_APBH_BASE           //0x80004000 *
MXS_PERFMON_BASE        //0x80006000 *
MXS_BCH_BASE            //0x8000A000 *
MXS_GPMI_BASE           //0x8000C000 *
MXS_SSP0_BASE           //0x80010000 *
MXS_SSP1_BASE           //0x80012000 *
MXS_SSP2_BASE           //0x80014000 *
MXS_SSP3_BASE           //0x80016000 *
MXS_PINCTRL_BASE        //0x80018000 Y
MXS_DIGCTL_BASE         //0x8001C000 Y
MXS_ETM_BASE            //0x80022000 ?
MXS_APBX_BASE           //0x80024000 *
MXS_DCP_BASE            //0x80028000 *
MXS_PXP_BASE            //0x8002A000 *
MXS_OCOTP_BASE          //0x8002C000 Y
MXS_AXI_AHB0_BASE       //0x8002E000 Y
MXS_LCDIF_BASE          //0x80030000 *
MXS_CAN0_BASE           //0x80032000 Y
MXS_CAN1_BASE           //0x80034000 Y
MXS_SIMDBG_BASE         //0x8003C000
MXS_SIMGPMISEL_BASE     //0x8003C200
MXS_SIMSSPSEL_BASE      //0x8003C300
MXS_SIMMEMSEL_BASE      //0x8003C400
MXS_GPIOMON_BASE        //0x8003C500
MXS_SIMENET_BASE        //0x8003C700
MXS_ARMJTAG_BASE        //0x8003C800
MXS_CLKCTRL_BASE        //0x80040000
MXS_SAIF0_BASE          //0x80042000 *
MXS_POWER_BASE          //0x80044000
MXS_SAIF1_BASE          //0x80046000 *
MXS_LRADC_BASE          //0x80050000 *
MXS_SPDIF_BASE          //0x80054000 *
MXS_RTC_BASE            //0x80056000 ?
MXS_I2C0_BASE           //0x80058000 *
MXS_I2C1_BASE           //0x8005A000 *
MXS_PWM_BASE            //0x80064000 *
MXS_TIMROT_BASE         //0x80068000 *
MXS_UARTAPP0_BASE       //0x8006A000 *
MXS_UARTAPP1_BASE       //0x8006C000 *
MXS_UARTAPP2_BASE       //0x8006E000 *
MXS_UARTAPP3_BASE       //0x80070000 *
MXS_UARTAPP4_BASE       //0x80072000 *
MXS_UARTDBG_BASE        //0x80074000 Y
MXS_USBPHY0_BASE        //0x8007C000 *
MXS_USBPHY1_BASE        //0x8007E000 *
MXS_USBCTRL0_BASE       //0x80080000
MXS_USBCTRL1_BASE       //0x80090000
MXS_DFLPT_BASE          //0x800C0000
MXS_DRAM_BASE           //0x800E0000
MXS_ENET0_BASE          //0x800F0000
MXS_ENET1_BASE          //0x800F4000
#endif

typedef struct {
	unsigned reg;
	char *name;
} reg_name;

#define REG_NAME(r)  { r,   #r}

reg_name reg_list[] = {
	REG_NAME(MXS_HSADC_BASE),
	REG_NAME(MXS_APBH_BASE),
	REG_NAME(MXS_PERFMON_BASE),
	REG_NAME(MXS_BCH_BASE),
	REG_NAME(MXS_GPMI_BASE),
	REG_NAME(MXS_SSP0_BASE),
	REG_NAME(MXS_SSP1_BASE),
	REG_NAME(MXS_SSP2_BASE),
	REG_NAME(MXS_SSP3_BASE),
	REG_NAME(MXS_APBX_BASE),
	REG_NAME(MXS_DCP_BASE),
	REG_NAME(MXS_PXP_BASE),
	REG_NAME(MXS_LCDIF_BASE),
	REG_NAME(MXS_SAIF0_BASE),
	REG_NAME(MXS_SAIF1_BASE),
	REG_NAME(MXS_LRADC_BASE),
	REG_NAME(MXS_SPDIF_BASE),
	REG_NAME(MXS_I2C0_BASE),
	REG_NAME(MXS_I2C1_BASE),
	REG_NAME(MXS_PWM_BASE),
	REG_NAME(MXS_TIMROT_BASE),
	REG_NAME(MXS_UARTAPP0_BASE),
	REG_NAME(MXS_UARTAPP1_BASE),
	REG_NAME(MXS_UARTAPP2_BASE),
	REG_NAME(MXS_UARTAPP3_BASE),
	REG_NAME(MXS_UARTAPP4_BASE),
	REG_NAME(MXS_USBPHY0_BASE),
	REG_NAME(MXS_USBPHY1_BASE),
};

#ifdef PRINT_SOME_REGISTERS
#define SFTRST (1<31)
#define CLKGATE (1<30)
static void mx28_chk_going(void)
{
int i;
unsigned v;
	for (i = 0; i < sizeof(reg_list)/sizeof(reg_list[0]); i++) {
		v = readl(reg_list[i].reg);
		if ((v & (SFTRST|CLKGATE)) == 0) {
			app_printf("Block %s is going\n\r", reg_list[i].name);
		}
	}
}
#endif


static void mx28_clocks_off(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

#define CLK_OFF(n,v) setbits_le32(&clkctrl_regs->n, v)

	CLK_OFF(hw_clkctrl_pll2ctrl0, CLKCTRL_PLL2CTRL0_CLKGATE);
	CLK_OFF(hw_clkctrl_ssp0,  CLKCTRL_SSP_CLKGATE);
	CLK_OFF(hw_clkctrl_ssp1,  CLKCTRL_SSP_CLKGATE);
	CLK_OFF(hw_clkctrl_ssp2,  CLKCTRL_SSP_CLKGATE);
	CLK_OFF(hw_clkctrl_ssp3,  CLKCTRL_SSP_CLKGATE);
	CLK_OFF(hw_clkctrl_gpmi,  CLKCTRL_GPMI_CLKGATE);
	CLK_OFF(hw_clkctrl_spdif, CLKCTRL_SPDIF_CLKGATE);
	CLK_OFF(hw_clkctrl_emi,   CLKCTRL_EMI_CLKGATE);
	CLK_OFF(hw_clkctrl_saif0, CLKCTRL_SAIF0_CLKGATE);
	CLK_OFF(hw_clkctrl_saif1, CLKCTRL_SAIF1_CLKGATE);
	CLK_OFF(hw_clkctrl_lcdif, CLKCTRL_DIS_LCDIF_CLKGATE);
	CLK_OFF(hw_clkctrl_etm,   CLKCTRL_ETM_CLKGATE);
	CLK_OFF(hw_clkctrl_enet,  CLKCTRL_ENET_DISABLE);
	CLK_OFF(hw_clkctrl_flexcan, CLKCTRL_FLEXCAN_STOP_CAN0| CLKCTRL_FLEXCAN_STOP_CAN1);
	clrbits_le32(&clkctrl_regs->hw_clkctrl_clkseq,
			CLKCTRL_CLKSEQ_BYPASS_DIS_LCDIF|
			CLKCTRL_CLKSEQ_BYPASS_ETM|
			CLKCTRL_CLKSEQ_BYPASS_EMI|
			CLKCTRL_CLKSEQ_BYPASS_SSP3|
			CLKCTRL_CLKSEQ_BYPASS_SSP2|
			CLKCTRL_CLKSEQ_BYPASS_SSP1|
			CLKCTRL_CLKSEQ_BYPASS_SSP0|
			CLKCTRL_CLKSEQ_BYPASS_GPMI|
			CLKCTRL_CLKSEQ_BYPASS_SAIF1|
			CLKCTRL_CLKSEQ_BYPASS_SAIF0 );
	// leave 32Khz timer on CLKCTRL_XTAL_TIMROT_CLK32K_GATE
	CLK_OFF(hw_clkctrl_xtal,  CLKCTRL_XTAL_UART_CLK_GATE|CLKCTRL_XTAL_PWM_CLK24M_GATE|CLKCTRL_XTAL_TIMROT_CLK32K_GATE);
	clrsetbits_le32(&clkctrl_regs->hw_clkctrl_hbus, 0x1f, 0x1f);
	clrsetbits_le32(&clkctrl_regs->hw_clkctrl_cpu, 0x3ff0000, 0x3ff0000);
	clrsetbits_le32(&clkctrl_regs->hw_clkctrl_xbus, 0x3ff, 0x3ff);
}

const iomux_cfg_t iomux_driver[] = {
MX28_PAD_LCD_D10__GPIO_1_10| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // red led
MX28_PAD_LCD_D15__GPIO_1_15| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // usb0 pwr en
MX28_PAD_LCD_D19__GPIO_1_19| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // zig pwr en
MX28_PAD_LCD_D20__GPIO_1_20| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // 5v_en
MX28_PAD_LCD_D21__GPIO_1_21| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // cell pwr en
MX28_PAD_LCD_D23__GPIO_1_23| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL),  // cell pwr off n
MX28_PAD_SSP0_SCK__GPIO_2_10| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL), // 3v3aux_en
MX28_PAD_SSP0_DETECT__GPIO_2_9| (MXS_PAD_3V3 | MXS_PAD_4MA | MXS_PAD_NOPULL), // Ignition
};

static void mx28_wgpio_setup(void)
{
// #define OUT_PINS_BANK_1 (1<<10)|(1<<15)|(1<<19)|(1<<20)|(1<<21)|(1<<23)
#define OUT_PINS_BANK_1 (1<<10)
#define OUT_PINS_BANK_2 (1<<10)
struct mx28_pinctrl_regs *pinctrl_regs =
	(struct mx28_pinctrl_regs *)MXS_PINCTRL_BASE;

	mxs_iomux_setup_multiple_pads(iomux_driver, ARRAY_SIZE(iomux_driver));
	clrbits_le32(&pinctrl_regs->hw_pinctrl_dout1, OUT_PINS_BANK_1);
	clrbits_le32(&pinctrl_regs->hw_pinctrl_dout2, OUT_PINS_BANK_2);
	clrsetbits_le32(&pinctrl_regs->hw_pinctrl_doe1, -1, OUT_PINS_BANK_1);
	clrsetbits_le32(&pinctrl_regs->hw_pinctrl_doe2, -1, OUT_PINS_BANK_2);
}

static void mx28_wgpio_init(void)
{
struct mx28_pinctrl_regs *pinctrl_regs =
	(struct mx28_pinctrl_regs *)MXS_PINCTRL_BASE;
iomux_cfg_t iomux;
int bank, pin;
int pinrange[] = { 29, 32, 28, 31, 32, 25, 26};
	for (bank = 0; bank < sizeof(pinrange)/sizeof(pinrange[0]); bank++) {
		for (pin = 0; pin < pinrange[bank]; pin++) {
			iomux = MXS_IOMUX_PAD(bank, pin, PAD_MUXSEL_GPIO, 0, 0, 0);
			mxs_iomux_setup_pad(iomux);
		}
	}
	clrbits_le32(&pinctrl_regs->hw_pinctrl_doe1, -1);
	clrbits_le32(&pinctrl_regs->hw_pinctrl_doe2, -1);
	clrbits_le32(&pinctrl_regs->hw_pinctrl_doe3, -1);
	clrbits_le32(&pinctrl_regs->hw_pinctrl_doe4, -1);
	mx28_wgpio_setup();
}

static void mx28_watch_disable(void)
{
#define RTC_PERSISTENT1_GENERAL_RTC_FORCE_UPDATER 0x80000000

        struct mx28_rtc_regs *rtc_regs = (struct mx28_rtc_regs *)MXS_RTC_BASE;

	writel(RTC_CTRL_WATCHDOGEN, &rtc_regs->hw_rtc_ctrl_clr);
        // setbits_le32(&rtc_regs->hw_rtc_ctrl_reg_clr, RTC_CTRL_WATCHDOGEN);
}

static void mx28_watch_enable(unsigned amount)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
	struct mx28_lcdif_regs *lcdif_regs =
		(struct mx28_lcdif_regs *)MXS_LCDIF_BASE;
	/*
	 * Shut down the LCD controller as it interferes with BootROM boot mode
	 * pads sampling.
	 */
	writel(LCDIF_CTRL_RUN, &lcdif_regs->hw_lcdif_ctrl_clr);

	/* Wait 1 uS before doing the actual watchdog reset */
	writel(amount, &rtc_regs->hw_rtc_watchdog);
	writel(RTC_CTRL_WATCHDOGEN, &rtc_regs->hw_rtc_ctrl_set);
}


// Spelling mistake in Freescale header file
#ifndef RTC_PERSISTENT0_XTAL24MHZ_PWRUP
#define RTC_PERSISTENT0_XTAL24MHZ_PWRUP RTC_PERSISTENT0_XTAL24KHZ_PWRUP
#endif

static void mx28_clear_alarm(void)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;

	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_set);
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_STALE_REGS_MASK) {}  ;
	clrsetbits_le32(&rtc_regs->hw_rtc_persistent0,
		/* CLEAR */
		RTC_PERSISTENT0_CLOCKSOURCE|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_LCK_SECS|
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		RTC_PERSISTENT0_XTAL32KHZ_PWRUP|
		RTC_PERSISTENT0_ALARM_WAKE|
		RTC_PERSISTENT0_LOWERBIAS_MASK|
		RTC_PERSISTENT0_DISABLE_PSWITCH|
		RTC_PERSISTENT0_AUTO_RESTART|
		0,
		/* SET */
		/* select one of 24Mhz or 32Khz */
		RTC_PERSISTENT0_LOWERBIAS_NOMINAL|
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		0);

        while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK) {} ;
}

/* The code calling mx28_alarm_time is commented out.
 * Keep here just for reference.
 */
#if 0
static void mx28_alarm_time(uint32_t t)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
	uint32_t v;

	v = readl(&rtc_regs->hw_rtc_seconds);
	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_set);
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_STALE_REGS_MASK) {}  ;

	writel(t+v, &rtc_regs->hw_rtc_rtc_alarm);
	clrsetbits_le32(&rtc_regs->hw_rtc_persistent0,
		/* CLEAR */
		RTC_PERSISTENT0_CLOCKSOURCE|
		RTC_PERSISTENT0_LOWERBIAS_MASK|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_ALARM_WAKE|
		RTC_PERSISTENT0_AUTO_RESTART|
		RTC_PERSISTENT0_DISABLE_PSWITCH|
		0,
		/* SET */
		RTC_PERSISTENT0_LOWERBIAS_NOMINAL|
		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		RTC_PERSISTENT0_AUTO_RESTART|
		/* select one of 24Mhz or 32Khz */
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		0);
        while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK) {} ;
}
#endif

static void mx28_alarm_auto(void)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;

	clrsetbits_le32(&rtc_regs->hw_rtc_persistent0,
		/* CLEAR */
		RTC_PERSISTENT0_CLOCKSOURCE|
		RTC_PERSISTENT0_LOWERBIAS_MASK|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_ALARM_WAKE|
		RTC_PERSISTENT0_AUTO_RESTART|
		RTC_PERSISTENT0_DISABLE_PSWITCH|
		0,
		/* SET */
		RTC_PERSISTENT0_LOWERBIAS_NOMINAL|
		RTC_PERSISTENT0_AUTO_RESTART|
 		RTC_PERSISTENT0_ALARM_EN|
		RTC_PERSISTENT0_ALARM_WAKE_EN|
		/* select one of 24Mhz or 32Khz */
		RTC_PERSISTENT0_XTAL24MHZ_PWRUP|
		0);
        while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK) {} ;
}

/* mx28_rtc_force is not used by any code.
 * Keep here for reference.
 */
#if 0
static void mx28_rtc_force(void)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_set);
	writel(RTC_CTRL_FORCE_UPDATE, &rtc_regs->hw_rtc_ctrl_clr);
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK)
		;
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_STALE_REGS_MASK)
		;
}
#endif

static inline void mx28_wait_ms(uint32_t ms)
{
uint32_t start;
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;

	start = readl(&rtc_regs->hw_rtc_milliseconds);
	while(readl(&rtc_regs->hw_rtc_milliseconds) - start < ms) {} ;
}

static uint32_t mx28_read_persist(int ind)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
	while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_STALE_REGS_MASK)
		;

	switch(ind) {
	    case 0:
		return readl(&rtc_regs->hw_rtc_persistent0);
	    case 1:
		return readl(&rtc_regs->hw_rtc_persistent1);
	    case 2:
		return readl(&rtc_regs->hw_rtc_persistent2);
	    case 3:
		return readl(&rtc_regs->hw_rtc_persistent3);
	    case 4:
		return readl(&rtc_regs->hw_rtc_persistent4);
	    case 5:
		return readl(&rtc_regs->hw_rtc_persistent5);
	}
	return 0;
}

static void mx28_write_persist(int ind, uint32_t v)
{
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;

	switch(ind) {
	    case 0:
		writel(v, &rtc_regs->hw_rtc_persistent0);
		break;
	    case 1:
		writel(v, &rtc_regs->hw_rtc_persistent1);
		break;
	    case 2:
		writel(v, &rtc_regs->hw_rtc_persistent2);
		break;
	    case 3:
		writel(v, &rtc_regs->hw_rtc_persistent3);
		break;
	    case 4:
		writel(v, &rtc_regs->hw_rtc_persistent4);
		break;
	    case 5:
		writel(v, &rtc_regs->hw_rtc_persistent5);
		break;
	}
        while (readl(&rtc_regs->hw_rtc_stat) & RTC_STAT_NEW_REGS_MASK) {} ;
}

static void mx28_led_blip(unsigned ms, unsigned cnt)
{
unsigned v;
	struct mx28_pinctrl_regs *pinctrl_regs =
		(struct mx28_pinctrl_regs *)MXS_PINCTRL_BASE;

	v = 0;
	while(1) {
		setbits_le32(&pinctrl_regs->hw_pinctrl_dout1,(1<<10));
		if (ms)
			mx28_wait_ms(ms);
		clrbits_le32(&pinctrl_regs->hw_pinctrl_dout1,(1<<10));
		v++;
		if (v >= cnt)
			break;
		if (ms)
			mx28_wait_ms(ms);
	}
}

#define COND_TIMER	0
#define COND_OFF	1
#define COND_ON		2
#define COND_INVALID	3

#define COND_VERSION	0x10000000		/* top three bits are version */
#define COND_VER_MASK	0xe0000000		/* top three bits are version */

static int mx28_gpio_wait(unsigned condition, unsigned cond_wait, unsigned ms, unsigned *remaining)
{
uint32_t v;
uint32_t start;
uint32_t led_start;
uint32_t cond_start = 0;
uint32_t cv;
int cond_started = 0;
int stop = 0;
	struct mx28_rtc_regs *rtc_regs =
		(struct mx28_rtc_regs *)MXS_RTC_BASE;
	struct mx28_pinctrl_regs *pinctrl_regs =
		(struct mx28_pinctrl_regs *)MXS_PINCTRL_BASE;

	led_start = start = readl(&rtc_regs->hw_rtc_milliseconds);
	while(1) {
		/* ignition pin configuration */
		/* falcon : gpio 2:9 */
		/* modcomm : gpio 2:16 */
#if (defined V_BOARD_nmc1000)
		v = readl(&pinctrl_regs->hw_pinctrl_din2) & (1<<16);
#else
		v = readl(&pinctrl_regs->hw_pinctrl_din2) & (1<<9);
#endif
		switch(condition) {
		    case COND_TIMER:
			break;
		    case COND_OFF:
		    case COND_ON:
			if (((condition == COND_OFF) && !v) ||
			    ((condition == COND_ON) && v)) {
				if (cond_wait) {
					if (!cond_started) {
						cond_start = readl(&rtc_regs->hw_rtc_milliseconds);
						cond_started = 1;
					}
				} else
					stop = 1;
			} else {
				cond_started = 0;
			}
			break;
		}
		cv = readl(&rtc_regs->hw_rtc_milliseconds);
		if (stop || (cond_started && cv - cond_start >= cond_wait)) {
			if (cv - start >= ms)
				*remaining = 0;
			else
				*remaining = (ms - (cv - start));
			return condition;
		}
		if (cv - start >= ms) {
			*remaining = 0;
			return COND_TIMER;
		}
		if (cv - led_start >= 10000) {
			led_start = cv;
			mx28_led_blip(0,0);
		}
	}
}

static void mx28_reset_chip(void)
{
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;

	writel(CLKCTRL_RESET_CHIP, &clkctrl_regs->hw_clkctrl_reset);
}

void mx28_power_init(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;
	uint32_t wait;
	uint32_t cond3;
	uint32_t code3;
	uint32_t cond4;
	uint32_t result;
	unsigned remaining;

	mx28_power_clock2xtal();
	mx28_power_clear_auto_restart();
	mx28_power_set_linreg();
	mx28_power_setup_5v_detect();

	mx28_setup_batt_detect();

	/* Charging capacitors, letting things settle.
	 * Measured threshold on nwl12 board is ~50ms,
	 * we double that. */
	early_delay(100*1000);

	mx28_power_configure_power_source();
#ifdef CONFIG_POWER_STATE_MACHINE
	mx28_spl_phase_1_power_state_machine();
#endif
	mx28_power_clock2pll();
#undef PRINT_SOME_REGISTERS
#ifdef PRINT_SOME_REGISTERS
	mx28_chk_going();
	pr_rtc();
	pr_clock();
	pr_power();
#endif
	wait = mx28_read_persist(5);
	cond3 = mx28_read_persist(3);
	cond4 = mx28_read_persist(4);
#undef JAMIE_DEBUG
#ifdef JAMIE_DEBUG
	wait =60;
	cond3 = (COND_OFF<<16)|0;
	cond4 = (COND_ON<<16)|10;
	app_printf("Setting wait, cond3 and cond4\n\r");
#endif
	if (wait) {
		mx28_write_persist(5, 0);

		/* GPIO's input and gpio mode */
#ifndef JAMIE_DEBUG
		mx28_wgpio_init();
#else
		mx28_wgpio_setup();
#endif

		/* MIN Power voodoo */
		mx28_setup_minpower();
		/* disable watchdog */
		mx28_watch_disable();
		/* stop the alarm in case it's going */
		mx28_clear_alarm();
		/* slow everything down and and turn off devices */
		wait *= 1000;
		remaining = 0;
		result = COND_INVALID;
		if (cond3) {
			mx28_write_persist(3, 0);
			code3 = (cond3 >> 16);
			if (code3 == COND_ON || code3 == COND_OFF) {
				result = mx28_gpio_wait(code3, ((unsigned)(cond3 & 0xfff)) * 10 , wait, &remaining);
				// app_printf("COND3 result %d, remain 0x%x\n\r", result, remaining);
				wait = remaining;
			}
		}
		if (result != COND_TIMER) {
			result = mx28_gpio_wait(cond4 >> 16, ((unsigned)(cond4 & 0xfff)) * 10 , wait, &remaining);
			// app_printf("COND4 result %d, remain 0x%x\n\r", result, remaining);
		}
		mx28_write_persist(4, COND_VERSION | (result<<16));
		//  mx28_led_blip(1,0);

		mx28_clocks_off();
		/* USB PLL OFF */
		mx28_pll0_off();

		mx28_reset_chip();
		// second chance
		mx28_alarm_auto();
		// mx28_alarm_time(1);

		while(1) {
			// power off disables watchdog
			mx28_powerdown();
			// third chance !
			mx28_watch_enable(1);
			asm("MOV R0, #0\n MCR p15, 0, r0, c7, c0, 4" : : : "r0");
		}
		/* never gets past here */
	} else {
		mx28_write_persist(4, COND_VERSION | (cond4 & ~COND_VER_MASK));	/* signal current version to booting system */
		mx28_enable_output_rail_protection();
		mx28_power_set_vddio(3300, 3150);
		mx28_power_set_vddd(1350, 1200);

		writel(POWER_CTRL_VDDD_BO_IRQ | POWER_CTRL_VDDA_BO_IRQ |
			POWER_CTRL_VDDIO_BO_IRQ | POWER_CTRL_VDD5V_DROOP_IRQ |
			POWER_CTRL_VBUS_VALID_IRQ | POWER_CTRL_BATT_BO_IRQ |
			POWER_CTRL_DCDC4P2_BO_IRQ, &power_regs->hw_power_ctrl_clr);
		writel(POWER_5VCTRL_PWDN_5VBRNOUT, &power_regs->hw_power_5vctrl_set);

		early_delay(1000);
	}
}

#ifdef	CONFIG_SPL_MX28_PSWITCH_WAIT
void mx28_power_wait_pswitch(void)
{
	struct mx28_power_regs *power_regs =
		(struct mx28_power_regs *)MXS_POWER_BASE;

	while (!(readl(&power_regs->hw_power_sts) & POWER_STS_PSWITCH_MASK))
		;
}
#endif
