/*
 * Freescale MX28 OCOTP fuse reading / burning
 *
 * Copyright (C) 2012 Ian Cullinan <ian.cullinan@netcommwireless.com>
 * on behalf of NetComm Wireless Limited
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <common.h>
#include <config.h>
#include <asm/io.h>
#include <asm/arch/imx-regs.h>
#include <asm/arch/sys_proto.h>

#include "mx28_init.h"

#define	MXS_OCOTP_MAX_TIMEOUT	1000000

void mx28_dump_fuses(void)
{
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	uint32_t data;
	uint32_t data2;

	app_uartbase(DUART_BASE);

	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_set);
	if (mx28_wait_mask_clr(&ocotp_regs->hw_ocotp_ctrl_reg, OCOTP_CTRL_BUSY,
				MXS_OCOTP_MAX_TIMEOUT)) {
		D("Can't open OCOTP fuse bank\r\n");
		return;
	}

	/* Get the MAC address */
	data = readl(&ocotp_regs->hw_ocotp_cust0);
	app_printf("MAC address: 00:60:64:%02x:%02x:%02x\r\n", (data >> 16) & 0xff, 
			(data >> 8) & 0xff, data & 0xff);

	/* Get the serial number */
	data = readl(&ocotp_regs->hw_ocotp_cust1);
	data2 = readl(&ocotp_regs->hw_ocotp_cust2);
	app_printf("Serial number: %07x%08x\r\n", data, data2);

	/* Get snextra */
	data = readl(&ocotp_regs->hw_ocotp_cust3);
	app_printf("SNextra: %08x\r\n", data);

	/* Close the fuse bank */
	writel(OCOTP_CTRL_RD_BANK_OPEN, &ocotp_regs->hw_ocotp_ctrl_clr);
}

#ifdef CONFIG_SPL_FUSEDATA_ADDR
/* from spl_power_init.c */
void mx28_power_set_vddio(uint32_t new_target, uint32_t new_brownout);

static void blow_fuse(uint32_t addr, uint32_t data) {
	struct mx28_clkctrl_regs *clkctrl_regs =
		(struct mx28_clkctrl_regs *)MXS_CLKCTRL_BASE;
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;

	D("Burning OCOTP fuse 0x%x with data 0x%x\r\n", addr, data);

	/* 1. Program HCLK to 24 MHz. OTP writes do not work at frequencies above 24 MHz. */
	/* Set CPU bypass - run off the 24MHz clock */
	writel(CLKCTRL_CLKSEQ_BYPASS_CPU, &clkctrl_regs->hw_clkctrl_clkseq_set);
	/* Set the HCLK ratio to divide-by-one */
	writel(0x01, &clkctrl_regs->hw_clkctrl_hbus_set);
	writel(0x1E, &clkctrl_regs->hw_clkctrl_hbus_clr);

	/* 2. Set the VDDIO voltage to 2.8 V */
	mx28_power_set_vddio(2800, 2700);

	/* 3. Check that HW_OCOTP_CTRL_BUSY and HW_OCOTP_CTRL_ERROR are clear. */
	if (readl(&ocotp_regs->hw_ocotp_ctrl_reg)
			& (OCOTP_CTRL_BUSY | OCOTP_CTRL_ERROR)) {
		D("OCOTP is busy or in error\r\n");
		return;
	}

	/* 4. Write the requested address to HW_OCOTP_CTRL_ADDR  and program the
	   unlock code into HW_OCOTP_CTRL_WR_UNLOCK. */
	writel(OCOTP_CTRL_WR_UNLOCK_KEY | (OCOTP_CTRL_ADDR_MASK & addr), 
			&ocotp_regs->hw_ocotp_ctrl_reg);

	/* 5. Write the data to HW_OCOTP_DATA. */
	writel(data, &ocotp_regs->hw_ocotp_data);

	/* 6. Once complete, the controller clears BUSY. */
	if (mx28_wait_mask_clr(&ocotp_regs->hw_ocotp_ctrl_reg, OCOTP_CTRL_BUSY,
				MXS_OCOTP_MAX_TIMEOUT)) {
		D("OCOTP write timed out\r\n");
		return;
	}

	/* 2us postamble (wait 4us just to be sure) */
	early_delay(4);
	
	/* Check for error */
	if (readl(&ocotp_regs->hw_ocotp_ctrl_reg) & OCOTP_CTRL_ERROR)
		D("Error writing OCOTP fuses\r\n");
}

struct spl_fusedata {
	uint32_t cust0;
	uint32_t cust1;
	uint32_t cust2;
	uint32_t cust3;
};

void mx28_blow_fuses(void) {
	struct mx28_ocotp_regs *ocotp_regs =
		(struct mx28_ocotp_regs *)MXS_OCOTP_BASE;
	struct spl_fusedata *fusedata =
		(struct spl_fusedata *)CONFIG_SPL_FUSEDATA_ADDR;

	/* Check to see if we actually have identity info to set */
	if (!(fusedata->cust0 && fusedata->cust1 && fusedata->cust2)) {
		D("No OCOTP fuse data provided - not blowing fuses\r\n");
		return;
	}

	/* Check to see that the fuses haven't already been blown */
	if (readl(&ocotp_regs->hw_ocotp_lock) & (OCOTP_LOCK_CUST0 |
			OCOTP_LOCK_CUST1 | OCOTP_LOCK_CUST2 | OCOTP_LOCK_CUST3)) {
		D("OCOTP fuses already blown - skipping\r\n");
		return;
	}

	D("Burning board identity into OCOTP fuses\r\n");
	blow_fuse(0, fusedata->cust0);
	blow_fuse(1, fusedata->cust1);
	blow_fuse(2, fusedata->cust2);
	blow_fuse(3, fusedata->cust3);
	/* Set the lock bits too */
	blow_fuse(0x10, (OCOTP_LOCK_CUST0 | OCOTP_LOCK_CUST1 | OCOTP_LOCK_CUST2
			| OCOTP_LOCK_CUST3));

	D("Board identity set\r\n");
}
#endif
